
;  PhotoServe 4.20

; ##### Disabling functions or modifying the PhotoServe script will earn you a permanent ban.  NO excuses. #####
; ##### If you have found a problem or are looking to make an enhancement please see a PhotoServe author or Channel Operator (in that order) before proceeding #####

/****************************************************************************************
* <b>PSST - PhotoServe Stats Module.</b>
* <p>
* Stats database and functions for updating it. PSCC Stats tab for presenting the data in
* text- and graphic-charts. Tracks daily transfer amounts at nick, trig, and file level.
* Also included is the old stats1 engine - eventually it will be phased out but currently
* we still use it for all the "Say -> Stats" etc messages.
* <p>
* <h3>Stats storage</h3>
* Stats are stored in hash tables, three per network (one for each type). Hash keys
* consist of one-letter prefix, followed by YYYY-MM-DD date, and nick or trig identifier
* (for nick and trig stats). The prefix letter is <code>'s'</code> for sends and
* <code>'g'</code> for gets. 
* <p>
* <code><b>File:</b> s2007-12-06 -> 66200572:9045:461:0:0:0:0:0:0:0:11891:Driller</code><br>
* <code><b>Nick:</b> s2007-12-03_mrys -> 14956302:9109:38:0:1:0:50:17:25:0:8181:mrys</code><br>
* <code><b>Trig:</b> s2007-12-03_DDCD2-> 67090760:33024:108:5:23:0:403:22:86:0:7663:leon^ps</code>
* <ol>
*  <li>Total number of bytes sent/get
*  <li>Total number of seconds spent on those sends/gets
*  <li>Number of !psget sends/gets that completed successfully
*  <li>Number of !psget sends/gets that failed to connect
*  <li>Number of !psget sends/gets that connected but later failed
*  <li>Number of !psget sends/gets that were cancelled with !pskill
*  <li>Number of !psserv sends/gets that completed successfully
*  <li>Number of !psserv sends/gets that failed to connect
*  <li>Number of !psserv sends/gets that connected but later failed
*  <li>Number of !psserv sends/gets that were cancelled with !pskill
*  <li>Record speed in Characters Per Second
*  <li>Nick who made the CPS record
* </ol>
* <p>
* Every day adds zero to two entries of filestats, one for gets, one for sends. If
* there are no gets/sends no entry needs to be created. The low storage requirement
* means we can keep file stats forever. For nick and trig stats we add one entry for
* each unique nick/trig - for the more active users this could mean 100s of entries
* per day so we cannot retain the data forever. Currently entries older than 30 days
* are purged daily, on the first hash save of the day.
* <p>
* All three stats hashes use the same format. This means we have some rather poinless
* data in the nick hash, and even more so in the trig hash. For example it's not very
* interesting to know how big % of your SNCD10 sends are failing. It is just a matter
* of wanting to write less code, both logging and querying.
* 
* <h3>Hash journal</h3>
* Every file transfer causes all three stats hashes to be modified. Because they can
* grow quite large we shouldn't write them to disk after every single transfer. On the
* other hand simply skipping some of the writes can lead to data loss in case of crashes.
* We solve this by maintaining a stats journal that logs all stats hash modifications.
* So for every file transferred we just append three lines in the journal, which is a
* lot better than doing three big /hsave's. Occasionally we flush the stats hashes to
* disk with /hsave and clear the journal.
* <p>
* To recover stats after a crash we load the latest stats hashes from disk, and then
* execute (aka replay) the journal entries one by one. Since the journal entries have
* been appended to the file in the same order that we have executed /hadd's for the
* hashes, the end result is the exact same state as existed before the crash. This 
* tolerance affords us to flush the hashes very rarely, currently once every 900 stat
* /hadd's (about 300 transferred files). The number may increase in the near future.
*
* <h3>Special hash keys</h3>
* Hash keys that begin and end with an underscore are special. <ul>
* <li> _IS_DIRTY_ - [all] Set whenever a hash is modified so we know which hashes need to be saved
* <li> _DATA_FORMAT_ - [all] Data version, allows for easier data format updates 
* <li> _STATS1_200x_SENDS_ - [file] Total gets recorded by stats1 in year 200x
* <li> _STATS1_200x_GETS_ - [file] Total sends recorded by stats1 in year 200x
* <li> _STATS2_START_ - [file] Since what day has stats2 data been recorded
* <li> _STATS1_START_ - [file] Since what day has stats1 data been recorded. This key exists even
* if stats1 data has never been recorded, with the same value is same as _STATS2_START_
* <li> _LAST_PRUNE_ - [nick &amp; trig] date the hash was last pruned of old entries
* </ul>
*
* <h3>Stats graphs</h3>
* Pserve.exe creates graphical charts with perl GD::Graph module. Pserve script just saves
* the current hashes and instructs pserve.exe to produce a chart with certain parameters.
* To reduce the latency of pserve.exe invocations the graph system uses the same communcation
* method first developed for PSCC TrigSetup tab. Same TCP/IP socket handles both TrigSetup
* and stats graph calls.
*
* <h3>Misc</h3>
* Global variables (settings):<ul>
* <li>%ps.whereis.system.up -- Current uptime record in seconds. I did not invent this variable name..
* </ul>
* Transient globals:<ul>
* <li>%ps_stats_unflushed -- number of unflushed updates across all stats hashes
* </ul>
* <h3>Bugs</h3>
* For some reason the TotalSeconds (second token) value for any day can become corrupt. 
* The corruption increases the value of TotalSeconds by 2^32 (=136 years). Knowing mirc
* it would not be surpising if <code>$send(n).secs</code> and <code>$get(n).secs</code>
* had bugs that cause them to return incorrect values, but no attempt to verify this has
* been made. The bug could be in the script too. 
* <p>
* <b>This bug has not been fixed</b> (can't be arsed ATM). We could clean up the past data
* and add safeguards to the logging aliases, but the lazy man's option is to fix the data
* during query. Just do <code>$calc($2 % 2^32)</code> when adding TotalSeconds values in
* <code>$PSST_SumFileStatsRange(...)</code> and other query aliases.
*/




/*
********************************************************************************************
*********************************************************************************************
*****************                                                        ********************
*****************       STATS DATABASE STORAGE AND MANIPULATION          ********************
*****************                                                        ********************
*********************************************************************************************
********************************************************************************************
*/


/****************************************************************************************
* Record file send, successfull or not, in to stats. Updates all three stats types.
* @param nick the user who the file was sent to
* @param trig collection trig ($null for misc files)
* @param method <code>!psserv</code> for DCC Server transfers, anything else counted as !psget
* @param sendnum The <code>'N'</code> for <code>$send(N)</code>
* @param pskilled pass true value if this send is about to die in response to !pskill
* @return void
*/
alias PSST_RecordSend {
  var %nick = $1, %trig = $2, %method = $3, %sendnum = $4, %pskilled = $5
  if (%nick == $me) return

  ; Determine end result of the send
  if ($send(%sendnum).done) {
    var %result = completed
  }
  elseif (%pskilled) {
    var %result = killed
  }
  elseif ($send(%sendnum).sent == $send(%sendnum).resume) {
    var %result = failed
  }
  else {
    var %result = died
  }

  ; Do sanity checks. The negative bytes thing happens when resume fails to connect
  var %net = $PS_Network
  if (%net == $null) {
    echo 3 -s PSST_RecordSend failed: Unable to determine network
    return
  }
  if ($send(%sendnum).sent < $send(%sendnum).resume) {
    ; echo 3 -s PSST_RecordSend failed: Will not log negative bytes
    return
  }

  ; Update stats 
  $PSST_RecordLog(File, $null, %net, s, %method, %nick, $send(%sendnum).sent, $send(%sendnum).resume, $send(%sendnum).secs, $send(%sendnum).cps, %result)
  $PSST_RecordLog(Nick, %nick, %net, s, %method, %nick, $send(%sendnum).sent, $send(%sendnum).resume, $send(%sendnum).secs, $send(%sendnum).cps, %result)
  if (%trig != $null) {
    $PSST_RecordLog(Trig, $upper(%trig), %net, s, %method, %nick, $send(%sendnum).sent, $send(%sendnum).resume, $send(%sendnum).secs, $send(%sendnum).cps, %result)
  }

  ; Add to stats1
  if (%result == completed) {
    $PS_FileStatsHash(Stats2, Send, $send(%sendnum).sent, $send(%sendnum).cps, %nick, $send(%sendnum).resume, $send(%sendnum).ip)
  }
  else {
    $PS_FileStatsHash(Stats2, Send, $send(%sendnum).sent, $send(%sendnum).cps, %nick, $send(%sendnum).resume, $send(%sendnum).ip).fail
  }
}



/****************************************************************************************
* Record file get, successfull or not, in to stats. Updates all three stats types.
* @param nick the user who sent me this file
* @param trig collection trig, $null for misc files
* @param method <code>!psserv</code> for DCC Server transfers, anything else counted as !psget
* @param getnum The <code>'N'</code> for <code>$get(N)</code>
* @param pskilled pass true value if this get died because of a !pskill
* @return void
*/
alias PSST_RecordGet {
  var %nick = $1, %trig = $2, %method = $3, %getnum = $4, %pskilled = $5
  if (%nick == $me) return

  ; Determine end result of the get
  var %result = $iif($get(%getnum).done, completed, failed)
  if ($get(%getnum).done) {
    var %result = completed
  }
  elseif (%pskilled) {
    var %result = killed
  }
  elseif ($get(%getnum).rcvd == $get(%getnum).resume) {
    var %result = failed
  }
  else {
    var %result = died
  }

  ; Do sanity checks. The negative bytes thing happens when resume fails to connect (at least for sends, gets??)
  var %net = $PS_Network
  if (%net == $null) {
    echo 3 -s PSST_RecordGet failed: Unable to determine network
    return
  }
  if ($get(%getnum).rcvd < $get(%getnum).resume) {
    ; echo 3 -s PSST_RecordGet failed: Will not log negative bytes
    return
  }

  ; Update stats 
  $PSST_RecordLog(File, $null, %net, g, %method, %nick, $get(%getnum).rcvd, $get(%getnum).resume, $get(%getnum).secs, $get(%getnum).cps, %result)
  $PSST_RecordLog(Nick, %nick, %net, g, %method, %nick, $get(%getnum).rcvd, $get(%getnum).resume, $get(%getnum).secs, $get(%getnum).cps, %result)
  if (%trig != $null) {
    $PSST_RecordLog(Trig, $upper(%trig), %net, g, %method, %nick, $get(%getnum).rcvd, $get(%getnum).resume, $get(%getnum).secs, $get(%getnum).cps, %result)
  }

  ; Add to stats1
  if (%result == completed) {
    $PS_FileStatsHash(Stats2, Get, $get(%getnum).rcvd, $get(%getnum).cps, %nick, $get(%getnum).resume, $get(%getnum).ip)
  }
  else {
    $PS_FileStatsHash(Stats2, Get, $get(%getnum).rcvd, $get(%getnum).cps, %nick, $get(%getnum).resume, $get(%getnum).ip).fail
  }
}




/****************************************************************************************
* Record get or send in one of stats hashes. All three stats hashes are manipulated with
* this single alias, and as a result use same data format. This is excessive especially
* for Trig stats (who cares about the send success rate or avg speed of one trigger),
* but it does result in better code reuse.
* @param stattype one of <code>'File', 'Trig', 'Nick'</code>
* @param keysuffix $null for File stats, for Trig and Nick stats include the nick/trig that is being logged.
* @param network network context of the transfer
* @param type <code>'s'</code> for send, <code>'g'</code> for get
* @param method <code>!psserv</code> for DCC Server transfers, anything else counted as !psget
* @param nick nick of the sending/receiving user
* @param rcvd bytes transferred
* @param resume resume point of the transfer (0 for non-resumed transfers)
* @param secs number of seconds the transfer took
* @param cps bytes/characters per second count
* @param result one of <code>'completed', 'failed', 'died', 'killed'</code>. Failed means
* "unable to connect", died means "connection established but lost during transfer"
* @return void
*/
alias -l PSST_RecordLog {
  var %stattype = $1, %keysuffix = $2, %network = $3, %type = $4, %method = $5, %nick = $6, %rcvd = $7, %resume = $8, %secs = $9, %cps = $10, %result = $11

  var %statshash = Ps $+ %stattype $+ Stats_ $+ %network
  if ($hget(%statshash) == $null) PSST_LoadStats %network
  var %hashkey = %type $+ $asctime(yyyy-mm-dd) $+ $iif(%keysuffix != $null, _ $+ %keysuffix, $null)
  tokenize 58 $hget(%statshash, %hashkey)

  var %bytes         = $calc($1 + %rcvd - %resume)
  var %seconds       = $calc($2 + %secs)
  var %psg_completed = $calc($3 + $iif(%method != !psserv && %result == completed, 1, 0))
  var %psg_failed    = $calc($4 + $iif(%method != !psserv && %result == failed, 1, 0))
  var %psg_died      = $calc($5 + $iif(%method != !psserv && %result == died, 1, 0))
  var %psg_killed    = $calc($6 + $iif(%method != !psserv && %result == killed, 1, 0))
  var %pss_completed = $calc($7 + $iif(%method == !psserv && %result == completed, 1, 0))
  var %pss_failed    = $calc($8 + $iif(%method == !psserv && %result == failed, 1, 0))
  var %pss_died      = $calc($9 + $iif(%method == !psserv && %result == died, 1, 0))
  var %pss_killed    = $calc($10 + $iif(%method == !psserv && %result == killed, 1, 0))
  var %maxcps        = $calc($11 + 0)
  var %maxnick       = $12
  if ((%cps > %maxcps) && (%secs > 1)) {
    %maxcps = %cps
    %maxnick = %nick
  }
  var %stats = $+(%bytes,:,%seconds,:,%psg_completed,:,%psg_failed,:,%psg_died,:,%psg_killed,:,%pss_completed,:,%pss_failed,:,%pss_died,:,%pss_killed,:,%maxcps,:,%maxnick)
  PSST_UpdateLog %statshash %hashkey %stats
  if ( %stattype == nick) .signal PSSTEvent_ $+ %stattype %keysuffix %type $calc(%rcvd - %resume)
  elseif (%stattype == trig) .signal PSSTEvent_ $+ %stattype %keysuffix %type $calc(%rcvd - %resume) %nick

}



/****************************************************************************************
* Load all stats hashes from disk to RAM. This function also check for the existance of
* stats journal. If the journal exists it means mirc has been shut uncleanly and the
* hashes need to be updated by replaying the journal entries (which this alias also does).
* @optparam network ensure that all required stats hashes for network exist (this is how
* new hahses are created when joining new networks for the first time) @return void
*/
alias -l PSST_LoadStats {
  var %network = $1

  ; Load stats hashes, three for each network
  var %i = $findfile($PS_StatsDir, Ps????Stats_*, 0, 1)
  while (%i > 0) {
    var %file  = $findfile($PS_StatsDir, Ps????Stats_*, %i, 1)
    var %hname = $remove($nopath(%file), .hsh)
    if ($hget(%hname) == $null) {
      hmake %hname
      hload %hname $qt(%file)
    }    
    dec %i
  }

  ; Replay journal entries to bring all hashes up to date
  if ($isfile($PS_StatsDir $+ PsStatsJournal.log)) {
    .fopen JOURNAL $PS_StatsDir $+ PsStatsJournal.log"
    while ((!$feof) && (!$ferr)) {
      var %line = $fread(JOURNAL)
      tokenize 32 %line
      if ($0 != 3) continue
      if ($1 == $chr(59)) continue
      if ($hget($1) == $null) continue
      hadd $1 $2 $3
      hadd $1 _IS_DIRTY_ 1
    }
    .fclose JOURNAL
    PSST_SaveStats
  }

  ; Create requested network specific hashes and migrate old data. Intetionally don't set _IS_DIRTY_
  ; if there is no stats1 data - no point in saving stats hashes that only have stats2 metadata.
  if (%network != $null) {
    var %hname = PsFileStats_ $+ %network
    if ($hget(%hname) == $null) { 
      hadd -m %hname _DATA_FORMAT_ 2
      hadd %hname _STATS1_START_ $asctime(yyyy-mm-dd)
      hadd %hname _STATS2_START_ $asctime(yyyy-mm-dd)
      PSST_MigrateStats1Data %network
    }
    var %hname = PsNickStats_ $+ %network
    if ($hget(%hname) == $null) hadd -m %hname _DATA_FORMAT_ 2
    var %hname = PsTrigStats_ $+ %network
    if ($hget(%hname) == $null) hadd -m %hname _DATA_FORMAT_ 2
  } 
}



/****************************************************************************************
* Migrate file stats data from old "stats1" system. Migration keeps all yearly stats,
* which we use for yearly stats (no surprises there) and all time total. Note that
* stats1 tracks avg speed differently from stats2. Stats2 sums the total <b>seconds</b>
* spent transfering files, and to get avg speed you divide by number of <b>bytes</b>
* tranfered. Stats1 sums the <b>CPS</b> of all transfers, and to get avg speed you
* divide it by the number of <b>files</b>.
* @param newhash name of the stats2 hash
* @param oldhash name of the stats1 hash
* @return void 
*/
alias -l PSST_MigrateStats1Data {
  var %network = $1
  var %newhash = PsFileStats_ $+ %network, %oldhash = PServeFileStats_ $+ %network

  ; Is there a stats1 database available? If not just leave
  if ($hget(%oldhash) == $null) {
    if ($isfile(PServeFileStats_ $+ %network $+ .hsh)) {
      $PS_FileStatsHashStatus(PSST_LoadStats, %oldhash, %network)
    }
    else {
      return
    }
  }

  ; Is there any worthwhile data in the stats1 database? Again just bail out if not
  if ((*:0:0:0:0:0:n/a iswm $hget(%oldhash, Get_History)) && (*:0:0:0:0:0:n/a iswm $hget(%oldhash, Send_History))) {
    return
  }

  ; Starting date (convert dd/mm/yyyy -> yyyy-mm-dd)
  tokenize 47 $hget(%oldhash, Stats_Date)
  if ($0 > 1) hadd %newhash _STATS1_START_ $3 $+ - $+ $2 $+ - $+ $1


  ; File stats (Crap@Year:FilesOK:Bytes:FilesFail:CpsSum:RecordSpeed:RecordNick -> Bytes:CpsSum:OK:RecSpeed:RecNick)
  var %i = $hget(%oldhash, 0).item
  while (%i > 0) {
    var %key = $lower($hget(%oldhash, %i).item)
    var %value = $hget(%oldhash, %key)
    tokenize 58 %value
    dec %i

    if ((Get_200? iswm %key) || (Send_200? iswm %key)) {
      var %newkey = _STATS1_ $+ $gettok(%key,2,95) $+ _ $+ $upper($gettok(%key,1,95)) $+ S_
      hadd %newhash %newkey $+($3,:,$5,:,$2,:,$6,:,$7)
    }
  }

  ; Some data was [probably] migrated, mark dirty and save
  hadd %newhash _IS_DIRTY_ 1
  PSST_SaveStats
}



/****************************************************************************************
* Set Key in stats hash Hname to Value. The new value is also logged in stats journal to
* prevent data loss during crashes. Once the journal grows over a predefined limit all
* hashes are flushed to disk and the journal is deleted. 
* @param hname name of the statshash to modify
* @param key name of the statshash key
* @param value the new value for the key
* @return void 
*/
alias -l PSST_UpdateLog {
  var %hname = $1, %key = $2, %value = $3

  write $PS_StatsDir $+ PsStatsJournal.log" %hname %key %value
  hadd %hname %key %value
  hadd %hname _IS_DIRTY_ 1

  set %ps_stats_unflushed $calc(%ps_stats_unflushed + 1)
  if (%ps_stats_unflushed > 900) { 
    .timerPSST_SaveStats -m 1 200 PSST_SaveStats
  }
}



/****************************************************************************************
* Flush all modified stats hashes to disk and delete journal file.
* @param void @return void
*/
alias -l PSST_SaveStats {
  ; Delete Nick & Trig stats entries that are older than 30 days.
  var %i = $hget(0)
  while (%i > 0) {
    var %hname = $hget(%i)
    if ((PsNickStats_* iswm %hname) || (PsTrigStats_* iswm %hname)) {
      if ($hget(%hname, _LAST_PRUNE_) != $asctime(yyyy-mm-dd)) { 
        var %j = 30
        while (%j < 180) {
          var %wcard = $asctime($calc($ctime - %j * 86400), ?yyyy-mm-dd_*)
          hdel -w %hname %wcard 
          inc %j
        }
        hadd %hname _LAST_PRUNE_ $asctime(yyyy-mm-dd)
        hadd %hname _IS_DIRTY_ 1
      }
    }
    dec %i
  }

  ; Save modified hashes
  var %i = $hget(0)
  while (%i > 0) {
    var %hname = $hget(%i)
    if ((Ps????Stats_* iswm %hname) && ($hget(%hname, _IS_DIRTY_) == 1)) {
      hdel %hname _IS_DIRTY_
      hsave %hname $PS_StatsDir $+ %hname $+ .hsh"
    }
    dec %i
  }
  .remove $PS_StatsDir $+ PsStatsJournal.log"
  unset %ps_stats_unflushed
}



/****************************************************************************************
* Return cumulative stats for specified range of dates. Returns the values in a space
* separated list (recommend using <code>/tokenize 32 %result</code> to split it),
* consisting of following fields:<ol>
* <li> bytes transfered
* <li> average CPS
* <li> psget success
* <li> psget noconnect
* <li> psget died
* <li> psget killed
* <li> psserv success
* <li> psserv noconnect
* <li> psserv died
* <li> psserv killed
* <li> record speed
* <li> record nick
* </ol>
* @param type either <code>'get'</code> or <code>'send'</code>
* @param startdate starting date in YYYY-MM-DD format
* @param days (non zero) number of days to traverse from the starting data, either negative or 
* positive number. Value of 7 means "startdate + next six days", value of -7 means "startdate
* + previous 6 days".
* @return tokens space separeted list (see numerated list above)
*/
alias PSST_SumFileStatsRange {
  var %network = $1, %type = $2, %startdate = $3, %days = $4
  %type = $left(%type, 1)
  var %hname = PsFileStats_ $+ %network
  if ($hget(%hname) == $null) PSST_LoadStats %network


  ; Convert yyyy-mm-dd date to unixtime, adjust down if days < 0
  var %ctime = $ctime(%startdate)
  if (%days < 0) {
    %days = $abs(%days)
    dec %ctime $calc(86400 * (%days - 1))
  }

  ; Init the results variables
  var %bytes_transfered  = 0
  var %total_seconds     = 0
  var %psget_success     = 0
  var %psget_noconnect   = 0
  var %psget_died        = 0
  var %psget_killed      = 0
  var %psserv_success    = 0
  var %psserv_noconnect  = 0
  var %psserv_died       = 0
  var %psserv_killed     = 0
  var %record_speed      = 0
  var %record_nick       = -

  ; Calculate stats
  while (%days > 0) {
    dec %days
    var %date = $calc(%ctime + 86400 * %days)
    var %stat = $hget(%hname, %type $+ $asctime(%date, yyyy-mm-dd))
    if (%stat == $null) continue
    tokenize 58 %stat

    inc %bytes_transfered $1
    inc %total_seconds    $calc($2 % 2^32)
    inc %psget_success    $3
    inc %psget_noconnect  $4
    inc %psget_died       $5
    inc %psget_killed     $6
    inc %psserv_success   $7
    inc %psserv_noconnect $8
    inc %psserv_died      $9
    inc %psserv_killed    $10
    if ($11 > %record_speed) {
      %record_speed = $11
      %record_nick = $12
    }
  }

  var %avg_speed = $int($calc(%bytes_transfered / %total_seconds))
  return %bytes_transfered %avg_speed %psget_success %psget_noconnect %psget_died %psget_killed %psserv_success %psserv_noconnect %psserv_died %psserv_killed %record_speed %record_nick
}




/****************************************************************************************
* Return cumulative stats for specified calendar year. Because the data comes partly from
* stats1 we can't provides as detailed breakdown as <code>PSST_SumFileStatsRange</code>,
* but arguably the fine details are not as interesting over longer periods. Returns 
* the values in space separated list (recommend using <code>/tokenize 32 %result</code>
* to split it), consisting of following fields:<ol>
* <li> bytes transfered
* <li> average CPS
* <li> files transfered
* <li> record speed
* <li> record nick
* </ol>
* @param type either <code>'get'</code> or <code>'send'</code>
* @return tokens space separeted list (see numerated list above)
*/
alias PSST_SumFileStatsYear {
  var %network = $1, %type = $2, %year = $3
  %type = $left(%type, 1)
  var %hname = PsFileStats_ $+ %network
  if ($hget(%hname) == $null) PSST_LoadStats %network

  ; Get old stats1 data (might or might not exist)
  if (%type == g) {
    var %stats1key = _STATS1_ $+ %year $+ _GETS_
  }
  else {
    var %stats1key = _STATS1_ $+ %year $+ _SENDS_
  }
  tokenize 58 $hget(%hname, %stats1key)

  ; Init results variables with stats1 data or empty values
  var %stats1_bytes      = $iif($1 != $null, $1, 0)
  var %stats1_avg        = $iif($2 != $null, $calc($2 / $3), 0)
  var %stats1_files      = $iif($3 != $null, $3, 0)
  var %record_speed      = $iif($4 != $null, $4, 0)
  var %record_nick       = $iif($5 != $null, $5, -)

  ; Get stats2 data (not available before 2007 - nice speedup if we skip if for earlier years)
  if (%year >= 2007) {
    var %date = %year $+ -01-01
    var %days = $iif($calc(%year % 4) == 0, 366, 365)
    tokenize 32 $PSST_SumFileStatsRange(%network, %type, %date, %days)
    var %stats2_bytes = $1
    var %stats2_avg   = $2
    var %stats2_files = $calc($3 + $7)
    if ($11 > %record_speed) {
      %record_speed = $11
      %record_nick  = $12
    }
  }

  ; Reconcile the difference in stats 1 & 2 methods of tracking avg speed
  var %files_transfered = $calc(%stats1_files + %stats2_files)
  var %stats1_weight = $calc(%stats1_files / %files_transfered)
  var %stats2_weight = $calc(%stats2_files / %files_transfered)
  var %avg_speed     = $calc(%stats1_avg * %stats1_weight) + $calc(%stats2_avg * %stats2_weight)

  :end
  return $calc(%stats1_bytes + %stats2_bytes) $int(%avg_speed) $calc(%stats1_files + %stats2_files) %record_speed %record_nick 
}



/****************************************************************************************
* Return cumulative all time stats for specified network. Returns data in same
* format as <code>PSST_SumFileStatsYear</code>.
* @param type either <code>'get'</code> or <code>'send'</code>
* @return tokens space separeted list (see <code>PSST_SumFileStatsYear</code>)
*/
alias PSST_SumFileStatsAll {
  var %network = $1, %type = $2
  var %hname = PsFileStats_ $+ %network
  if ($hget(%hname) == $null) PSST_LoadStats %network

  ; Init results variables with empty values
  var %bytes_transfered  = 0
  var %avg_speed         = 0
  var %files_transfered  = 0
  var %record_speed      = 0
  var %record_nick       = -

  ; Loop from starting year to current year and sum the returned values
  var %year = $left($hget(%hname, _STATS1_START_), 4)
  var %years = 0
  while (%year <= $asctime(yyyy)) {
    tokenize 32 $PSST_SumFileStatsYear(%network, %type, %year)
    inc %bytes_transfered $1
    inc %avg_speed        $2
    inc %files_transfered $3
    if ($3 > 0) {
      inc %years
    }
    if ($4 > %record_speed) {
      %record_speed = $4
      %record_nick  = $5
    }
    inc %year
  }

  ; Divide summed-up avg speed by number of years (sucks but stats1 & stats2 diffs mean we have no choice)
  %avg_speed = $int($calc(%avg_speed / %years))

  return %bytes_transfered %avg_speed %files_transfered %record_speed %record_nick
}




/****************************************************************************************
* Return average send speed over last %min_files filesends. 
* @param min_files go back in stats so that the avg speed number is calculated over 
* AT LEAST this many file sends
* @return float average sends speed in KB/s, with one decimal precision
*/
alias PSST_GetAvgUpSpeed {
  var %network = $1, %min_files = $2
  var %hname = PsFileStats_ $+ %network

  ; Loop back from today until stats cover at least %min_files (but no more than 200 days)
  var %i = 0, %total_seconds = 0, %bytes_transfered = 0, %files_success = 0
  while (%i < 200) {
    var %date = $calc($ctime - 86400 * %i)
    inc %i    
    var %stat = $hget(%hname, s $+ $asctime(%date, yyyy-mm-dd))
    if (%stat == $null) continue

    tokenize 58 %stat
    inc %bytes_transfered $1
    inc %total_seconds    $calc($2 % 2^32)
    inc %files_success    $3
    inc %files_success    $7
    if (%files_success >= %min_files) break
  }

  return $round($calc(%bytes_transfered / %total_seconds / 1024), 1)
}



/****************************************************************************************
* Return get/send ratio as signed decimal number. The %get and %send params can be raw
* byte counts, or (as in the case of servinfo formatting) precalced ratio numbers.
* Very large value are clamped to +999 and -999. Values above and 100 and below -100
* do not have any decimals, they are rounded to nearest integer.
* @prop color add Green|Orange|Red color formatting to the ratio
* @return ratio signed decimal number with
*/
alias PSST_CalcRatio {
  var %gets = $1, %sends = $2
  if (%gets == $null) %gets = 0
  if (%sends == $null) %sends = 0

  ; Calculate and format ratio
  if ((%gets == 0) && (%sends != 0)) {
    var %ratio = +999
  }
  else if ((%gets != 0) && (%sends == 0)) {
    var %ratio = -999
  }
  else if (%gets <= %sends) {
    var %ratio = + $+ $round($calc(%sends / %gets), 2)
  }
  else if (%gets > %sends) {
    var %ratio = - $+ $round($calc(%gets / %sends), 2)
  }
  else {
    var %ratio = +1.00
  }

  ; Normalize huge values
  if (%ratio > 999) {
    %ratio = +999
  }
  if (%ratio < -999) {
    %ratio = -999
  }

  ; Two decimals if (100 > ratio > -100), no decimals if not
  if ($regex(%ratio,/[\-\+]\d\d?$/))     %ratio = %ratio $+ .00
  if ($regex(%ratio,/[\-\+]\d\d?\.\d$/)) %ratio = %ratio $+ 0
  if (%ratio > 100)  %ratio = + $+ $round(%ratio, 0)
  if (%ratio < -100) %ratio = $round(%ratio, 0)

  ; Add color if requested
  if ($prop == color) {
    if (%ratio < -5) {
      %ratio = 04 $+ %ratio
    }
    else if (%ratio < -3) {
      %ratio = 07 $+ %ratio
    }
    else {
      %ratio = 03 $+ %ratio
    }
  }

  return %ratio
}




;; Save stats on exit
on *:exit: {
  PSST_SaveStats
}




/*
********************************************************************************************
*********************************************************************************************
*****************                                                        ********************
*****************                    SAY/ECHO STATS                      ********************
*****************                                                        ********************
*********************************************************************************************
********************************************************************************************
*/


/****************************************************************************************
* Show one line of stats for today, last x days, current month/year, or complete history.
* @param command_string what to do with the generated stats line. Example <code>/echo</code>, <code>/say $chan</code>, etc
* @param time_string- one of ( <code>today</code>      | <code>yesterday</code> | <code>last N days</code> |
*                             <code>this month</code> | <code>this year</code> | <code>history</code> )
* @return ... Usually <code>$null</code>, unless <code>%command_string</code> is "<code>return</code>"
*/
alias PSST_SayStats {
  var %command_string = $1, %time_string = $2-
  var %network = $PS_Network


  ; Calculate stats date range (not for History and This Year)
  if (%time_string == today) {
    var %date = $asctime(yyyy-mm-dd)
    var %days = 1
  }
  elseif (%time_string == yesterday) {
    var %date = $asctime($calc($ctime - 86400), yyyy-mm-dd)
    var %days = 1
  }
  elseif ($regex(%time_string, /last\s+(\d+)\s+days/i) ) {
    var %date = $asctime(yyyy-mm-dd)
    var %days = - $+ $regml(1)
  }
  elseif (%time_string == this month) {
    var %date = $asctime(yyyy-mm-01)
    var %days = 31
  }

  ; Fetch values from stats DB
  if (%time_string == history) {
    tokenize 32 $PSST_SumFileStatsAll(%network, get)
    var %get_bytes = $1, %get_avg = $2, %get_files = $3, %get_record_speed = $4, %get_record_nick = $5
    tokenize 32 $PSST_SumFileStatsAll(%network, send)
    var %send_bytes = $1, %send_avg = $2, %send_files = $3, %send_record_speed = $4, %send_record_nick = $5
    %time_string = Since $hget(PsFileStats_ $+ %network, _STATS1_START_)
  }
  elseif (%time_string == this year) {
    tokenize 32 $PSST_SumFileStatsYear(%network, get, $asctime(yyyy))
    var %get_bytes = $1, %get_avg = $2, %get_files = $3, %get_record_speed = $4, %get_record_nick = $5
    tokenize 32 $PSST_SumFileStatsYear(%network, send, $asctime(yyyy))
    var %send_bytes = $1, %send_avg = $2, %send_files = $3, %send_record_speed = $4, %send_record_nick = $5
  }
  elseif (%date != $null) {
    tokenize 32 $PSST_SumFileStatsRange(%network, get, %date, %days)
    var %get_bytes = $1, %get_avg = $2, %get_files = $calc($3 + $7), %get_record_speed = $11, %get_record_nick = $12
    tokenize 32 $PSST_SumFileStatsRange(%network, send, %date, %days)
    var %send_bytes = $1, %send_avg = $2, %send_files = $calc($3 + $7), %send_record_speed = $11, %send_record_nick = $12
  }
  else {
    return
  }


  ; Format the simple stuff for display
  var %files = 10Files(03 $+ %get_files $+ 10/04 $+ %send_files $+ 10)
  var %sizes = 10Sizes(03 $+ $PS_FileSize(%get_bytes,1) $+ 10/04 $+ $PS_FileSize(%send_bytes,1) $+ 10)
  var %averages = 10Avg KB/s(03 $+ $round($calc(%get_avg  / 1024), 1) $+ 10/04 $+ $round($calc(%send_avg / 1024), 1) $+ 10)


  ; Format get/send record speeds and nicks for display
  var %records = 10Max KB/s(
  if (%get_record_speed > 0) {
    %records = %records $+ 03 $+ %get_record_nick 07 $+ $round($calc(%get_record_speed  / 1024), 1)
  }
  else {
    %records = %records $+ 03-
  }
  if (%send_record_speed > 0) {
    %records = %records $+ 10/04 $+ %send_record_nick 07 $+ $round($calc(%send_record_speed / 1024), 1) $+ 10)
  }
  else {
    %records = %records $+ 10/04-10)
  }

  ; Calculate and format ratio & ratio message
  var %ratio = $PSST_CalcRatio(%get_bytes, %send_bytes)
  var %ratiomsg =  $+ %ps.whereis.eqmsg
  if (%ratio < 1) %ratiomsg =  $+ %ps.whereis.getmsg
  if (%ratio > 1) %ratiomsg =  $+ %ps.whereis.sendmsg
  var %ratio = 10Ratio( $+ $PSST_CalcRatio(%get_bytes, %send_bytes).color $+ 10)

  %command_string 12(03Get12/04Send12)07 %time_string %files %sizes %records %averages %ratio %ratiomsg
}



/****************************************************************************************
* Show N lines of stats for last N days/months/years, plus one line for totals
* The lines contain a little less data than given by <code>PSST_SayStats(..)</code>,
* trying to avoid line wrapping so the results would be more table-like and easier to read.
* @param command_string what to do with the generated stats lines
* @param n how many days/months/years to go back
* @param breakdown one of ( <code>day</code> | <code>month</code> | <code>year</code> )
* @return void
*/
alias PSST_SayStatsMultiple {
  var %command_string = $1, %n = $2, %breakdown = $3
  var %network = $PS_Network

  var %total_send_bytes = 0, %total_send_files = 0, %total_get_bytes = 0, %total_get_files = 0, %total_n = 0

  ; Calc N times (once for each requested day/month/year)
  while (%n > 0) {
    dec %n

    ; Calculate stats date range. Couldn't think of a cleaner way to calculate months due to the varying length
    if (%breakdown == day) {
      var %date = $asctime($calc($ctime - 86400 * %n), yyyy-mm-dd)
      var %days = 1
      var %time_string = %date
    }
    else if (%breakdown == month) {
      var %i = 1
      var %months_delta = 0
      var %date = $asctime(yyyy-mm-01)
      ; Do 15-day jumps backwards, and count how many times the month changes
      while (%months_delta < %n) {
        var %try_date = $asctime($calc($ctime - 86400 * 15 * %i), yyyy-mm-01)
        if (%try_date != %date) {
          %date = %try_date
          inc %months_delta        
        }
        inc %i
      }
      tokenize $asc(-) %date
      var %days = $PSST_DaysInMonth($2, $1)
      var %time_string = $asctime($ctime(%date), mmm yyyy)
    }
    else if (%breakdown == year) {
      var %year = $calc($asctime(yyyy) - %n)
      var %time_string = Year %year
    }

    ; Fetch values from stats DB
    if (%breakdown == year) {
      tokenize 32 $PSST_SumFileStatsYear(%network, get, %year)
      var %get_bytes = $1, %get_avg = $2, %get_files = $3
      tokenize 32 $PSST_SumFileStatsYear(%network, send, %year)
      var %send_bytes = $1, %send_avg = $2, %send_files = $3
    }    
    else {
      tokenize 32 $PSST_SumFileStatsRange(%network, get, %date, %days)
      var %get_bytes = $1, %get_avg = $2, %get_files = $calc($3 + $7)
      tokenize 32 $PSST_SumFileStatsRange(%network, send, %date, %days)
      var %send_bytes = $1, %send_avg = $2, %send_files = $calc($3 + $7)
    }

    ; Never begin the display with "empty" stats. If this would be first stats line displayed and there are no
    ; send and no gets, skip to the next day/month/year.
    if ((%total_n == 0) && (%send_bytes == 0) && (%get_bytes == 0)) continue
    inc %total_n    

    ; Add to totals
    inc %total_send_bytes %send_bytes
    inc %total_send_files %send_files
    inc %total_get_bytes %get_bytes
    inc %total_get_files %get_files

    ; Format values for display and display using %command_string
    var %files = 10Files(03 $+ %get_files $+ 10/04 $+ %send_files $+ 10)
    var %sizes = 10Sizes(03 $+ $PS_FileSize(%get_bytes,1) $+ 10/04 $+ $PS_FileSize(%send_bytes,1) $+ 10)
    var %averages = 10Avg KB/s(03 $+ $round($calc(%get_avg  / 1024), 1) $+ 10/04 $+ $round($calc(%send_avg / 1024), 1) $+ 10)
    var %ratio = 10Ratio( $+ $PSST_CalcRatio(%get_bytes, %send_bytes).color $+ 10)
    %command_string 12(03Get12/04Send12)07 %time_string %files %sizes %averages %ratio
  }

  ; Say totals
  var %files = 10Files(03 $+ %total_get_files $+ 10/04 $+ %total_send_files $+ 10)
  var %sizes = 10Sizes(03 $+ $PS_FileSize(%total_get_bytes,1) $+ 10/04 $+ $PS_FileSize(%total_send_bytes,1) $+ 10)
  var %ratio = 10Ratio( $+ $PSST_CalcRatio(%total_get_bytes, %total_send_bytes).color $+ 10)
  %command_string 12(03Get12/04Send12)07 Total for last %total_n %breakdown $+ s %files %sizes %ratio
}




/*
********************************************************************************************
*********************************************************************************************
*****************                                                        ********************
*****************                    PSCC STATS TAB                      ********************
*****************                                                        ********************
*********************************************************************************************
********************************************************************************************
*/


/****************************************************************************************
* Create MDX listview and select default checkbox/radiobutton options. The buttons need
* to be selected here because the first PSST_DrawGraph runs before PSCC_PopulateTab1400
* (currently PSCC_Resize happens before PSCC_Popuplate, and resize trigges DrawGraph).
* @param void @return void
*/
alias PSCC_CreateTab1400 {
  var %dname = $1

  ; Set initial view to Alltime with Yearly breakdown, draw everything
  did -c %dname 1410,1415,1420,1421,1422,1423
  did -b %dname 1425,1426

  $PSCC_MakeListview(%dname,1443, single showsel rowselect report, 90:1 40:2 42:3, +l Nick / Trig, +r MBs, +r Files)
}


/****************************************************************************************
* Populate user interface components with data and select initial graph type and style.
* Also ensures that logs for this network are loaded so we don't need to check it later on.
* @param void @return void
*/
alias PSCC_PopulateTab1400 {
  var %dname = $1

  ; Add months and select current month
  var %i = 1
  while (%i <= 12) {
    did -a %dname 1426 %i
    if (%i == $asctime(m)) {
      did -c %dname 1426 %i
    }
    inc %i
  }

  ; Add years and select current year. Also loads network log
  var %network = $PS_Network
  PSST_LoadStats %network
  var %addyear = $left($hget(PsFileStats_ $+ %network, _STATS2_START_), 4)
  while (%addyear <= $asctime(yyyy)) {
    did -a %dname 1425 %addyear
    inc %addyear
  }
  did -c %dname 1425 $did(%dname, 1425).lines

  ; Nick / Trig stats selection combobox
  did -a %dname 1442 Nicks I leeched (today)
  did -a %dname 1442 Nicks I leeched (yesterday)
  did -a %dname 1442 Nicks I leeched (last 7 days)
  did -a %dname 1442 Nicks I leeched (last 30 days)
  did -a %dname 1442 -------------------------------------
  did -a %dname 1442 Nicks I sent to (today)
  did -a %dname 1442 Nicks I sent to (yesterday)
  did -a %dname 1442 Nicks I sent to (last 7 days)
  did -a %dname 1442 Nicks I sent to (last 30 days)
  did -a %dname 1442 -------------------------------------
  did -a %dname 1442 Trigs I leeched (today)
  did -a %dname 1442 Trigs I leeched (yesterday)
  did -a %dname 1442 Trigs I leeched (last 7 days)
  did -a %dname 1442 Trigs I leeched (last 30 days)
  did -a %dname 1442 -------------------------------------
  did -a %dname 1442 Trigs I sent (today)
  did -a %dname 1442 Trigs I sent (yesterday)
  did -a %dname 1442 Trigs I sent (last 7 days)
  did -a %dname 1442 Trigs I sent (last 30 days)

  ; Draw with default options
  .timerPSST_DrawGraph -m 1 100 PSST_DrawGraph %dname

  ; Uptime stats
  PSST_RefreshUptimeDialog
  .timerPSST_RefreshUptimeDialog 0 60 PSST_RefreshUptimeDialog %dname
}


/****************************************************************************************
* Resize and redraw graph when dialog size changes. The graph image needs to be hidden
* during the resize or it bleeds/tears when reducing dialog size. Downside of this is
* some pretty bad flickering but that's only temporary whereas the corruption is permanent.
* @param void @return void
*/
alias PSCC_ResizeTab1400 {
  var %w = $1, %h = $2, %dname = $3
  did -h %dname 1499
  PS_Mdx MoveControl %dname 1499 * * $calc(%w - 250) *
  did -v %dname 1499
  .timerPSST_DrawGraph -m 1 100 PSST_DrawGraph %dname

  PS_Mdx MoveControl %dname 1440 * * * $calc(%h - 306)
  PS_Mdx MoveControl %dname 1443 * * * $calc(%h - 380)
  PS_Mdx MoveControl %dname 1450 * $calc(%h - 85) * *
  PS_Mdx MoveControl %dname 1451 * $calc(%h - 60) * *
  PS_Mdx MoveControl %dname 1452 * $calc(%h - 60) * *
  PS_Mdx MoveControl %dname 1453 * $calc(%h - 40) * *
  PS_Mdx MoveControl %dname 1454 * $calc(%h - 40) * *

  ; For some reason these texts disappear during resize so we must for a redraw 
  did -h %dname 1451,1452,1453,1454
  did -v %dname 1451,1452,1453,1454
}


;; Set graph period = Last 30 days
on *:DIALOG:PS_Dlg_AX:sclick:1410: {
  did -ue $dname 1415,1416,1417,1418
  did -b $dname 1417,1418,1425,1426
  did -c $dname 1415
  PSST_DrawGraph $dname
}

;; Set graph period = Month
on *:DIALOG:PS_Dlg_AX:sclick:1411: {
  did -ue $dname 1415,1416,1417,1418
  did -b $dname 1417,1418
  did -e $dname 1425,1426
  did -c $dname 1415
  PSST_DrawGraph $dname
}

;; Set graph period = Year
on *:DIALOG:PS_Dlg_AX:sclick:1412: {
  did -ue $dname 1415,1416,1417,1418
  did -e $dname 1425
  did -c $dname 1417
  did -b $dname 1418,1426
  PSST_DrawGraph $dname
}

;; Set graph period = All time
on *:DIALOG:PS_Dlg_AX:sclick:1413: {
  did -ue $dname 1415,1416,1417,1418
  did -c $dname 1418
  did -b $dname 1425,1426
  PSST_DrawGraph $dname
}

;; Change breakdow, drawing options, or year/month
on *:DIALOG:PS_Dlg_AX:sclick:1415-1430: {
  PSST_DrawGraph $dname
}

;; Click graph image to go to full screen mode
on *:DIALOG:PS_Dlg_AX:sclick:1499: {
  PSST_DrawGraph $dname $true
}





/****************************************************************************************
* Dispatch pserve.exe to draw a graph with requested attributes.
* @param fullscreen pass true value to view the graph in full screen mode @return void
*/
alias -l PSST_DrawGraph {
  var %fullscreen = $2
  var %network = $PS_Network
  if (%network == $null) return

  ; Parse the dialog form in to easier to read variables
  var %dname  = $1
  var %width  = $calc($dialog(%dname).cw - 250)
  var %height = 324
  var %year   = $did(%dname,1425).seltext
  var %mon    = $did(%dname,1426).seltext
  if ($did(%dname,1415).state) var %breakdown = day
  if ($did(%dname,1416).state) var %breakdown = week
  if ($did(%dname,1417).state) var %breakdown = month
  if ($did(%dname,1418).state) var %breakdown = year
  if ($did(%dname,1410).state) var %period = 30d
  if ($did(%dname,1411).state) var %period = month
  if ($did(%dname,1412).state) var %period = year
  if ($did(%dname,1413).state) var %period = alltime
  var %draw_points = $did(%dname, 1420).state
  var %draw_values = $did(%dname, 1421).state
  var %draw_gets   = $did(%dname, 1422).state
  var %draw_sends  = $did(%dname, 1423).state

  ; Set starting date and number of days to full calendar periods (execept for 30d option)
  if (%period == 30d) {
    var %ctime = $calc($ctime - 86400 * 29)
    var %days = 30
  }
  if (%period == month) {
    var %start = %year $+ - $+ %mon $+ -01
    var %ctime = $ctime(%start)
    if ($asctime($calc(%ctime + 86400 * 31), m) == %mon)     var %days = 31
    elseif ($asctime($calc(%ctime + 86400 * 30), m) == %mon) var %days = 30
    elseif ($asctime($calc(%ctime + 86400 * 29), m) == %mon) var %days = 29
    else                                                     var %days = 28
  }
  if (%period == year) {
    var %start = %year $+ -01-01
    var %ctime = $ctime(%start)
    if ($asctime($calc(%ctime + 86400 * 366), yyyy) == %year) var %days = 366
    else                                                      var %days = 365
  }
  if (%period == alltime) {
    var %start = $hget(PsFileStats_ $+ %network, _STATS2_START_)
    var %ctime = $ctime(%start)
    var %days  = $int($calc(($ctime - %ctime) / 86400 + 1))
  }

  PSST_SaveStats
  ;echo -s DrawGraph %width %height $PS_Network %draw_points %draw_values %draw_gets %draw_sends %breakdown %ctime %days

  if (%fullscreen) {
    $PSS_DisplayFullscreen($PS_TmpDir $+ StatsGraph.png")
    %width  = $window(@PSS_FS_Display).w
    %height = $int($calc(%width / 16 * 7))
    $PSEXE_WriteConnection(DrawGraph %width %height $PS_Network %draw_points %draw_values %draw_gets %draw_sends %breakdown %ctime %days, PSS_DisplayFullscreen $PS_TmpDir $+ StatsGraph.png)
  }
  else {
    $PSEXE_WriteConnection(DrawGraph %width %height $PS_Network %draw_points %draw_values %draw_gets %draw_sends %breakdown %ctime %days, PSST_LoadGraph %dname)
  }
}

/****************************************************************************************
* Load the graph that pserve.exe created. This function is public because it's run by
* callback from another module. @param void @return void
*/
alias PSST_LoadGraph {
  var %dname = $1
  if ($isfile($PS_TmpDir $+ StatsGraph.png")) {
    did -g %dname 1499 $PS_TmpDir $+ StatsGraph.png"
  }
}

;; Change Nick/Trig Stats selection
on *:DIALOG:PS_Dlg_AX:sclick:1442: {
  if ($PS_Network == $null) return
  did -r $dname 1443

  ; Determine timeframe. Same options for all type of stats
  var %sel = $did($did).sel
  if ($calc(%sel % 5) == 1) {
    var %date  = $asctime(yyyy-mm-dd)
    var %range = 1
  }
  elseif ($calc(%sel % 5) == 2) {
    var %date  = $asctime($calc($ctime - 86400), yyyy-mm-dd)
    var %range = 1
  }
  elseif ($calc(%sel % 5) == 3) {
    var %date  = $asctime($calc($ctime - 86400 * 6), yyyy-mm-dd)
    var %range = 7
  }
  elseif ($calc(%sel % 5) == 4) {
    var %date  = $asctime($calc($ctime - 86400 * 29), yyyy-mm-dd)
    var %range = 30
  }
  else {
    return
  }

  ; Determine type: nick/trig and get/send
  if (%sel < 5) {
    var %stat = nick
    var %type = get
  }
  elseif (%sel < 10) {
    var %stat = nick
    var %type = send
  }
  elseif (%sel < 15) {
    var %stat = trig
    var %type = get
  }
  elseif (%sel < 20) {
    var %stat = trig
    var %type = send
  }

  $PSST_PopulateNickTrigStats($dname, $PS_Network, %stat, %type, %date, %range)
}


/****************************************************************************************
* Populate Nick / Trig stats MDX listview. The list is sorted from biggest transfer
* amount to the smallest.
* @param stat either <code>'Nick'</code> or <code>'Trig'</code>
* @param type either <code>'get'</code> or <code>'send'</code>
* @param startdate starting date YYYY-MM-DD format
* @param days how long period from starting day forward to calculate. Must be positive number.
* @return void
*/
alias -l PSST_PopulateNickTrigStats {
  var %dname = $1, %network = $2, %stat = $3, %type = $4, %startdate = $5, %days = $6
  %type = $left(%type, 1)
  var %hname = Ps $+ %stat $+ Stats_ $+ %network

  ; Create temporary "type and dates of interest" hash
  hfree -w PsStatsDOI
  hmake PsStatsDOI
  var %i = 0
  while (%i < %days) {
    var %ctime = $calc($ctime(%startdate) + 86400 * %i)
    var %date  = $asctime(%ctime, yyyy-mm-dd)
    hadd PsStatsDOI %type $+ %date 1
    inc %i
  }

  ; Iterate entire PsNickStats hash and sum "interesting" data in temp result hash
  hfree -w PsStatsRes
  hmake PsStatsRes
  var %i = $hget(%hname, 0).item
  while (%i > 0) {
    var %key = $hget(%hname, %i).item
    var %date = $left(%key, 11)
    if ($hget(PsStatsDOI, %date) == 1) {
      var %stat = $hget(%hname, %key)
      tokenize 58 %stat 
      var %bytes = $1, %files = $3 + $7

      var %nick = $right(%key, -12)
      var %stat = $hget(PsStatsRes, %nick)
      tokenize 58 %stat 
      %bytes = $calc(%bytes + $1)
      %files = $calc(%files + $2)
      hadd PsStatsRes %nick $+(%bytes,:,%files)
    }
    dec %i
  }

  ; Dump result hash in to sorted @window, thus sorting it by bytes
  window -hls @PsStatsSort
  while ($hget(PsStatsRes, 0).item > 0) {
    var %nick  = $hget(PsStatsRes, 1).item
    var %files = $gettok($hget(PsStatsRes, %nick), 2, 58)
    var %bytes = $gettok($hget(PsStatsRes, %nick), 1, 58)
    while ($len(%bytes) < 12) %bytes = 0 $+ %bytes
    hdel PsStatsRes %nick
    aline @PsStatsSort  %bytes %files %nick
  }

  ; Finally, move the sorted data from the @window to the MDX listview
  var %i = $line(@PsStatsSort, 0)
  while (%i > 0) {
    tokenize 32 $line(@PsStatsSort, %i)
    var %megs  = $calc($1 / 1024 / 1024)
    if (%megs < 1000) {
      %megs = $round(%megs, 1)
      if ($left($right(%megs,2),1) != .) %megs = %megs $+ .0
    }
    else {
      %megs  = $round(%megs, 0)
    }
    var %files = $2
    var %nick  = $3
    did -a %dname 1443 $+(+ 0 0 0 %nick, $chr(9),+ 0 0 0 %megs, $chr(9), + 0 0 0 %files)
    dec %i
  }

  ; Remove temp hashes and windows
  hfree PsStatsRes
  hfree PsStatsDOI
  window -c @PsStatsSort
}




/*
********************************************************************************************
*********************************************************************************************
*********************************                        ************************************
*********************************   UPTIME STATS STUFF   ************************************
*********************************                        ************************************
*********************************************************************************************
********************************************************************************************
*/


/****************************************************************************************
* Refresh current uptime and uptime record in PSCC Stats tab.
* @param void @return void
*/
alias -l PSST_RefreshUptimeDialog {
  var %dname = $1
  if ($dialog(%dname) == $null) {
    .timerPSST_RefreshUptimeDialog off
    return
  }
  PSST_SaveUptimeStats
  did -ra %dname 1452 $duration(%PS.whereis.system.up, 2)
  did -ra %dname 1454 $uptime(system, 2)
}


/****************************************************************************************
* Return current uptime record in human readable format
* @param void @return string "2wks 4days 6hrs"
*/
alias PSST_GetUptimeRecord {
  PSST_SaveUptimeStats
  return $duration(%PS.whereis.system.up, 2)
}


/****************************************************************************************
* Refresh stored uptime record (if better than old record).
* @param void @return void
*/
alias -l PSST_SaveUptimeStats {
  if ($uptime(system, 3) > %PS.whereis.system.up) {
    set %PS.whereis.system.up $uptime(system, 3)
    saveini
  }
}

;; Trigger PSST_SaveUptimeStats on quit
on *:QUIT: {
  PSST_SaveUptimeStats
}

;; Trigger PSST_SaveUptimeStats on exit
on *:EXIT: {
  PSST_SaveUptimeStats
}



/****************************************************************************************
* Return number of Days for the month
* @param month number 1..12
* @optparam year speficy year if you want leap year handling
* @return days number 28..31
*/
alias PSST_DaysInMonth {
  var %month = $1, %year = $2
  if (%month == 2) {
    if (%year == $null) return 28
    if ($calc(%year % 4) != 0) return 28
    if ($calc(%year % 100) != 0) return 29
    if ($calc(%year % 400) != 0) return 28
    return 29
  }
  if (%month <= 7) return $calc( 30 + %month % 2)
  else return $calc( 31 - %month % 2)
}












































































/*
********************************************************************************************
*********************************************************************************************
*****************                                                        ********************
*****************   OLD STATS1 ALIASES - STILL IN USE SO DO NOT REMOVE   ********************
*****************                                                        ********************
*********************************************************************************************
********************************************************************************************
*/
; day@year   : 2  totalfiles   :  3 totalbytes   :  4 fail   :  5 avgcps   :  6 reccps   :  7 recnick
alias PS_FileStatsHash {
  var %ps_a = PS_FileStatsHash
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = identifier
  ; $2 = send / get
  ; $3 = file size
  ; $4 = cps
  ; $5 = nick
  ; $6 = resume
  ; $7 = ip

  if ($5 == $me) return

  ; calculate date for stats
  var %ps_date = $asctime(dddd) $+ $PS_Week(%ps_a,$asctime(yyyy-mm-dd)) $+ $chr(64) $+ $asctime(yyyy)
  var %ps_network = $ps_network(%ps_a)

  ; set values for successfull transfer
  var %ps_send = 1
  var %ps_fail = 0
  var %ps_bytes = $calc($3 - $6)

  ; if transfer failed set values accordingly
  if ($prop == fail) {
    var %ps_send = 0
    var %ps_fail = 1
    var %ps_bytes = $calc($3 - $6)
  }

  if (%ps_bytes <= 0) return

  ; check status of hash system
  var %ps_statshashtable = PServeFileStats_ $+ %ps_network
  $PS_FileStatsHashStatus(%ps_a,%ps_statshashtable,%ps_network)
  $PS_FileStatsHashStatus(%ps_a,PServeFileStats_PhotoServe,PhotoServe)

  ; write data to hash file
  $PS_FileStatsHashWrite(%ps_a,%ps_statshashtable,$2, $+ %ps_date $+ : $+ %ps_send $+ : $+ %ps_bytes $+ : $+ %ps_fail $+ : $+ $4 $+ : $+ $5 $+ ,%ps_network)
  if ($prop != fail) $PS_FileStatsHashWrite(%ps_a,PServeFileStats_PhotoServe,$2, $+ %ps_date $+ : $+ %ps_send $+ : $+ %ps_bytes $+ : $+ %ps_fail $+ : $+ $4 $+ : $+ $5 $+ ,PhotoServe)
}



alias PS_FileStatsHashWrite {
  var %ps_a = PS_FileStatsHashWrite
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = calling alias
  ; $2 = table
  ; $3 = send / get
  ; $4 = day@year   :   inc send   :   totalbytes   :   inc fail   :   cps   :   nick
  ; $5 = network

  var %ps_table = $2
  var %ps_section = $3
  var %ps_data = $4
  var %ps_network = $5

  var %ps_count = 1
  while (%ps_count <= 5) {

    ; determine which item to process
    if (%ps_count = 1) var %ps_item = today
    elseif (%ps_count = 2) var %ps_item = $PS_Week(%ps_a,$asctime(yyyy-mm-dd))
    elseif (%ps_count = 3) var %ps_item = -m $+ $asctime(mm)
    elseif (%ps_count = 4) var %ps_item = $asctime(yyyy)
    elseif (%ps_count = 5) var %ps_item = History

    ; get the curent string from the stats file
    ; $1 = day@year   : $2 = totalfiles   :  $3 = totalbytes   :  $4 = fail   :  $5 = avgcps   :  $6 = reccps   :  $7 = recnick
    var %ps_read = $PS_FileStatsHashGet(%ps_a,%ps_table,%ps_section,%ps_item)

    if (%ps_count == 1) {
      if (n/a isin %ps_read) {
        hadd %ps_table %ps_section $+ _ $+ PreviousDay $PS_FileStatsHashGet(%ps_a,%ps_table,%ps_section,today,save)
      }
    }

    ;  Calculate new values for read string $2-5 with $2-5 from data ( $2 = total files , $3 = total bytes , $4 = fail , $5 = avg cps )
    var %ps_c = 2
    while (%ps_c <= 5) {
      var %ps_tokcur = $gettok(%ps_read,%ps_c,58)
      var %ps_tokadd = $gettok(%ps_data,%ps_c,58)
      var %ps_tokcalc = $calc(%ps_tokcur + %ps_tokadd)
      if ((%ps_count == 1) && (%ps_c == 2)) {
        if (. !isin $calc(%ps_tokcalc / 25)) {
          var %ps_save = save
        }
      }
      var %ps_read = $puttok(%ps_read,%ps_tokcalc,%ps_c,58)
      inc %ps_c
    }

    ;  IF $6 from read trig < $5 from data write new record info ( $6 = reccps , $7 = recnick )
    var %ps_tokcur = $gettok(%ps_read,6,58)
    var %ps_tokadd = $gettok(%ps_data,5,58)
    var %ps_toknick = $gettok(%ps_data,6,58)
    if (%ps_tokadd > %ps_tokcur) {
      var %ps_read = $puttok(%ps_read,%ps_tokadd,6,58)
      var %ps_read = $puttok(%ps_read,%ps_toknick,7,58)
    }

    :write
    hadd %ps_table %ps_section $+ _ $+ %ps_item %ps_read
    if (%ps_table == PServeFileStats_PhotoServe) goto save
    inc %ps_count
  }

  :save
  if (%ps_save == save) {
    $PS_FileStatsHashSave(%ps_a,%ps_table,%ps_network)
  }
}



alias PS_FileStatsHashGet {
  var %ps_a = PS_FileStatsHashGet
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = alias
  ; $2 = table
  ; $3 = get / send / stats
  ; $4 = today / -w# / -m# / year / history / date / PreviousDay
  ; $5 = yes / null = [multiple dates to read] / save

  if ($2 == $null) return
  if ($3 == $null) return

  ; check status of hash system
  var %ps_network = $ps_network(%ps_a)
  var %ps_statshashtable = PServeFileStats_ $+ %ps_network
  $PS_FileStatsHashStatus(%ps_a,%ps_statshashtable,%ps_network)
  $PS_FileStatsHashStatus(%ps_a,PServeFileStats_PhotoServe,PhotoServe)

  var %ps_read = $hget($2,$+($3,_,$4))

  if (%ps_read != $null) {
    var %ps_year = $gettok(%ps_read,1,58)

    ; return value for today
    if ($4 == today) {
      if ($5 == save) return %ps_read
      if ($+($asctime(dddd),$PS_Week(%ps_a,$asctime(yyyy-mm-dd)),$CHR(64),$asctime(yyyy)) != %ps_year) goto zero
    }

    ; return value for PreviousDay
    if (($4 == PreviousDay) && ($+($asctime($calc($ctime - 86400),dddd),$PS_Week(%ps_a,$asctime($calc($ctime - 86400),yyyy-mm-dd)),$CHR(64),$asctime(yyyy)) != %ps_year)) goto zero

    ; return value for all others
    if ($5 != yes) {
      if ($asctime(yyyy) !isin %ps_year) {
        if (-w isin $4) goto zero
        if (-m isin $4) goto zero
        if ($4 == year) goto zero
      }
    }

    return %ps_read
  }

  if ($3 == stats) {
    var %ps_date = $date
    hadd $2 $+($3,_,$4) %ps_date
    return %ps_date
  }

  :zero
  var %ps_calc = 0
  if ($4 == PreviousDay) var %ps_calc = 86400
  return $+($asctime($calc($ctime - %ps_calc),dddd),$PS_Week(%ps_a,$asctime($calc($ctime - %ps_calc),yyyy-mm-dd)),$CHR(64),$asctime(yyyy),:,0,:,0,:,0,:,0,:,0,:,n/a)
}



alias PS_FileStatsHashFile {
  var %ps_a = PS_FileStatsHashFile
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  return $PS_SettingsDir(%ps_a) $+ PServeFileStats_ $+ $2 $+ .hsh $+ "
}



alias PS_FileStatsHashFileInit {
  var %ps_a = PS_FileStatsHashFileInit
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = calling function
  ; $2 = file
  ; $3 = network

  if ($3 != PhotoServe) {
    $PS_Write(%ps_a,$2,Stats_Date)
    $PS_Write(%ps_a,$2,$date)
  }

  var %ps_date = $asctime(dddd) $+ $PS_Week(%ps_a,$asctime(yyyy-mm-dd)) $+ $chr(64) $+ $asctime(yyyy)

  var %ps_count = 1
  while (%ps_count <= 2) {
    if (%ps_count == 1) var %ps_section = Get
    elseif (%ps_count == 2) var %ps_section = Send

    var %ps_c = 1
    while (%ps_c <= 6) {
      if (%ps_c = 1) var %ps_item = today
      elseif (%ps_c = 2) var %ps_item = $PS_Week(%ps_a,$asctime(yyyy-mm-dd))
      elseif (%ps_c = 3) var %ps_item = -m $+ $asctime(mm)
      elseif (%ps_c = 4) var %ps_item = $asctime(yyyy)
      elseif (%ps_c = 5) var %ps_item = History
      elseif (%ps_c = 6) var %ps_item = PreviousDay
      $PS_Write(%ps_a,$2,%ps_section $+ _ $+ %ps_item)
      $PS_Write(%ps_a,$2,%ps_date $+ : $+ 0 $+ : $+ 0 $+ : $+ 0 $+ : $+ 0 $+ : $+ 0 $+ : $+ n/a)
      if ($3 == PhotoServe) goto next
      inc %ps_c
    }

    :next
    inc %ps_count
  }

}



alias PS_FileStatsHashStatus {
  var %ps_a = PS_FileStatsHashStatus
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = calling function
  ; $2 = table
  ; $3 = network

  ; if hash table does not exist read from file
  if ($hget($2) == $null) $PS_FileStatsHashBuild(%ps_a,$2,$3)

  ; if hash file does not exist write file from hash table
  if (!$isfile($PS_FileStatsHashFile(%ps_a,$3))) $PS_FileStatsHashSave(%ps_a,$2,$3)
}



alias PS_FileStatsHashBuild {
  var %ps_a = PS_FileStatsHashBuild
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = calling function
  ; $2 = table
  ; $3 = network

  ; if .hsh file does not exist build it.
  var %ps_statshashfile = $PS_FileStatsHashFile(%ps_a,$3)
  if (!$isfile(%ps_statshashfile)) $PS_FileStatsHashFileInit(%ps_a,%ps_statshashfile,$3)

  ; clear, make and load the hash table
  hfree -w $2
  hmake $2 200
  hload $2 %ps_statshashfile
}



alias PS_FileStatsHashSave {
  var %ps_a = PS_FileStatsHashSave
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = calling function
  ; $2 = table
  ; $3 = network

  if ($2 != $null) {
    var %ps_network = $3
    var %ps_statshashtable = $2
  }
  else {
    var %ps_network = $ps_network(%ps_a)
    var %ps_statshashtable = PServeFileStats_ $+ %ps_network
  }

  if ($hget(%ps_statshashtable,0).item > 0) {
    hsave -o %ps_statshashtable $PS_FileStatsHashFile(%ps_a,%ps_network)
    hsave -o PServeFileStats_PhotoServe $PS_FileStatsHashFile(%ps_a,PhotoServe)
  }

}



alias PS_FileStatsHashReload {
  var %ps_a = PS_FileStatsHashReload
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  var %ps_network = $ps_network
  var %ps_table = PServeFileStats_ $+ %ps_network

  ; Reload hash file or find new hash file to load.
  var %ps_where = $?!="Yes = Reload existing trigger $crlf No = Specify new file"
  if (%ps_where == $true) var %ps_statshashfile = $PS_FileStatsHashFile(%ps_a,%ps_network)
  else if (!$isfile(%ps_statshashfile)) $sfile($PS_scriptdir $+ *_*.hsh)
  if (!$isfile(%ps_statshashfile)) return

  ; clear, make and load the hash table
  hfree -w %ps_table
  hmake %ps_table 200
  hload %ps_table %ps_statshashfile
}



alias PS_Week {
  var %ps_a = PS_Week
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = calling function
  ; $2 = 2004-03-06

  var %checkdate = $ctime($2 0:00:00)
  var %yearbegin = $ctime($left($2,4) $+ -01-01 0:00:00)
  var %yearoffset = $PS_Week_Offset(PS_Week,$2)

  var %period = $int($calc((%checkdate - %yearbegin + %yearoffset - 86400 + $daylight) / 302400 / 2))

  return -w $+ $iif($len(%period) == 1,0 $+ %period,%period)
}



alias PS_Week_Offset {
  var %ps_a = PS_Week_Offset
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = calling function
  ; $2 = 2004-03-06

  var %yearbegin = $left($2,4) $+ -01-01
  var %startday = $asctime($ctime(%yearbegin),ddd)
  if (%startday isin "Mon") return 86400
  if (%startday isin "Tue") return $calc(86400 * 2)
  if (%startday isin "Wed") return $calc(86400 * 3)
  if (%startday isin "Thu") return $calc(86400 * 4)
  if (%startday isin "Fri") return $calc(86400 * 5)
  if (%startday isin "Sat") return $calc(86400 * 6)
  if (%startday isin "Sun") return $calc(86400 * 7)
}
















alias PS_DialogGigDays {
  var %ps_a = PS_DialogGigDays
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = month

  if ($1 != $null) var %ps_month = $1
  else var %ps_month = $asctime(mm)

  ; return # of days in specified month
  if (%ps_month isin 01,03.05,07,08,10,12) return 31
  elseif (%ps_month isin 04,06,09,11) return 30
  elseif (%ps_month == 02) {
    if (. isin $calc($asctime(yyyy) / 4)) return 28
    else return 29
  }
}

alias PS_DialogGigLimitInit {
  var %ps_a = PS_DialogGigLimitInit
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = calling function
  ; $2 = $dname

  ; init the dropdown daily global bandwidth limit list
  var %ps_count = 1
  while (%ps_count <= 20) {
    did -a $2 384 $calc(20 * %ps_count + 80) MB
    inc %ps_count
  }
  var %ps_count = 1
  while (%ps_count <= 45) {
    did -a $2 384 $calc(100 * %ps_count + 400) MB
    inc %ps_count
  }
  var %ps_count = 5
  while (%ps_count <= 20) {
    did -a $2 384 $calc(1000 * %ps_count) MB
    inc %ps_count
  }
  var %ps_count = 2
  while (%ps_count <= 5) {
    did -a $2 384 $calc(20000 * %ps_count) MB
    inc %ps_count
  }

  ; get limit value from settings file
  var %ps_netstatusfile = $PS_SettingsDir(PS_DialogGigLimitInit) $+ PServeSettings_ $+ PhotoServe $+ .ini"
  var %ps_meglimit = $readini(%ps_netstatusfile,n,#Status,#MegLimit)

  if (%ps_meglimit != $null) {

    ; enable meg limit things
    did -c $2 382

    ; select the set gig limit from the list
    did -c $2 384 $findtok($didtok($2,384,35),%ps_meglimit,35)

    ; populate limit values in dialog
    did -rae $2 393 $calc($gettok(%ps_meglimit,1,32) / 2) MB
    did -rae $2 394 $round($calc($gettok(%ps_meglimit,1,32) * $PS_DialogGigDays / 1024),2) GB
  }
  else {
    did -b $2 384
    did -rh $2 393,394
  }

}

alias PS_DialogGigCurrentCount {
  var %ps_a = PS_DialogGigCurrentCount
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  ; $1 = alias
  ; $2 = get / send
  ; $3 = return

  var %ps_curget = 0
  var %ps_cursend = 0
  var %ps_curtotal = 0

  var %ps_curget = $gettok($PS_FileStatsHashGet(PS_DialogGigCurrentCount,PServeFileStats_PhotoServe,Get,Today),3,58)
  var %ps_cursend = $gettok($PS_FileStatsHashGet(PS_DialogGigCurrentCount,PServeFileStats_PhotoServe,Send,Today),3,58)
  var %ps_curtotal = $calc(%ps_curget + %ps_cursend)

  var %ps_netstatusfile = $PS_SettingsDir(PS_DialogGigCurrentCount) $+ PServeSettings_ $+ PhotoServe $+ .ini"
  var %ps_meglimit = $gettok($readini(%ps_netstatusfile,n,#Status,#MegLimit),1,32)
  var %ps_meglimit = $calc($calc(1024 * 1024 * %ps_meglimit) / 2)

  var %dname = PS_Dlg_AX
  if ($dialog(%dname) != $null) {
    did -ra %dname 386 $PS_FileSize(%ps_curget,2)
    did -ra %dname 388 $PS_FileSize(%ps_cursend,2)
    did -ra %dname 391 $PS_FileSize(%ps_curtotal,2)
  }

  if (%ps_meglimit == $null) return
  if (%ps_meglimit == 0) return

  if ($2 == get) {
    if (%ps_curget > %ps_meglimit) {
      return off %ps_meglimit
    }
  }

  if ($2 == send) {
    if (%ps_cursend > %ps_meglimit) {
      return off %ps_meglimit
    }
  }

}





























alias PS_DialogAutoTimeInit {
  var %ps_a = PS_DialogAutoTimeInit
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  var %ps_count = 0
  while (%ps_count <= 23) {
    did -a $1 $2 $+ $chr(44) $+ $3 %ps_count $+ : $+ 00
    did -a $1 $2 %ps_count $+ : $+ 30
    inc %ps_count
  }
  var %dname = $1
  did -d %dname 377 1
  did -c %dname $2 $findtok($didtok($1,$2,35),$chr(32),35)
  did -c %dname $3 $findtok($didtok($1,$3,35),$chr(32),35)
}

alias PS_AutoOnOffWrite {
  var %ps_a = PS_AutoOnOffWrite
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  var %ps_network = $did($dname,361).seltext
  var %ps_on_hr = $gettok($1,1,58)
  var %ps_on_min = $gettok($1,2,58)
  var %ps_dur_hr = $gettok($2,1,58)
  var %ps_dur_min = $gettok($2,2,58)
  var %ps_off_min = $calc(%ps_on_min + %ps_dur_min)
  if (%ps_off_min >= 60) {
    var %ps_off_min = $calc(%ps_off_min - 60)
    var %ps_off_hr = 1
  }
  if (%ps_off_min == 0) var %ps_off_min = 00
  var %ps_off_hr = $calc(%ps_on_hr + %ps_dur_hr + %ps_off_hr)
  if (%ps_off_hr >= 24) {
    var %ps_off_hr = $calc(%ps_off_hr - 24)
  }
  var %ps_off_time = %ps_off_hr $+ $chr(58) $+ %ps_off_min
  did -ram PS_Dlg_AX 379 %ps_off_time
  var %ps_netstatusfile = $PS_SettingsDir(PS_AutoOnOffWrite) $+ PServeSettings_ $+ %ps_network $+ .ini"
  $PS_WriteIni(PS_AutoOnOffWrite,%ps_netstatusfile,#Status,#TimeOn,$1)
  $PS_WriteIni(PS_AutoOnOffWrite,%ps_netstatusfile,#Status,#TimeDuration,$2)
  $PS_WriteIni(PS_AutoOnOffWrite,%ps_netstatusfile,#Status,#TimeOff,%ps_off_time)
  $PS_AutoOnOffTimer
}

alias PS_AutoOnOffTimer {
  var %ps_a = PS_AutoOnOffTimer
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  var %ps_temp_total = $scon(0)
  var %ps_temp_count = 1
  while (%ps_temp_count <= %ps_temp_total) {
    scon -t1 %ps_temp_count
    var %ps_network = $ps_network(PS_AutoOnOffTimer)
    var %ps_netstatusfile = $PS_SettingsDir(PS_AutoOnOffTimer) $+ PServeSettings_ $+ %ps_network $+ .ini"
    var %ps_on_time = $readini(%ps_netstatusfile,n,#Status,#TimeOn)
    var %ps_off_time = $readini(%ps_netstatusfile,n,#Status,#TimeOff)
    if ((%ps_on_time != $null) && (%ps_off_time != $null)) {
      $PS_AutoOnOffMessage(%ps_on_time,%ps_off_time)
      .timer $+ PS_AutoOn_ $+ %ps_network %ps_on_time 1 0 PS_AutoOnOffProc ON
      .timer $+ PS_AutoOff_ $+ %ps_network %ps_off_time 1 0 PS_AutoOnOffProc OFF %ps_on_time %ps_off_time
    }
    else {
      .timer $+ PS_AutoOn_ $+ %ps_network off
      .timer $+ PS_AutoOff_ $+ %ps_network off
    }
    scon -r
    inc %ps_temp_count
  }

  .timer $+ PS_AutoOn_GigCount 0:00 1 0 $!PS_DialogGigCurrentCount(PS_AutoOnOffTimer1)
  $PS_DialogGigCurrentCount(PS_AutoOnOffTimer2)

}

alias PS_AutoOnOffProc {
  var %ps_a = PS_AutoOnOffProc
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  if ($1 == ON) {
    $PS_SetStatus(Timer,ON)
    $PS_AutoOnOffMessage(ON)
  }
  if ($1 == OFF) {
    $PS_SetStatus(Timer,OFF,Auto,Auto On $2 / Off $3)
    $PS_AutoOnOffMessage(OFF)
  }
  .timer $+ PS_AutoOnOff 1 90 PS_AutoOnOffTimer
}

alias PS_AutoOnOffMessage {
  var %ps_a = PS_AutoOnOffMessage
  if (%ps_debug_alias != $null) echo $ps_h -st * %ps_a * $1-

  var %ps_network = $ps_network(PS_AutoOnOffMessage)
  if ($2 != $null) {
    echo $ps_n -st --> Photoserve will turn 3ON daily at $+ $ps_kh $1
    echo $ps_n -st --> Photoserve will turn 4OFF daily at $+ $ps_kh $2
  }
  else {
    if ($1 == ON) echo $ps_n -st --> Photoserve 3Enabled on $+ $ps_kh %ps_network by $+ $ps_kh Auto On/Off timer
    if ($1 == OFF) echo $ps_n -st --> Photoserve 4Disabled on $+ $ps_kh %ps_network by $+ $ps_kh Auto On/Off timer
  }
}
