
;  PhotoServe 4.20

; ##### Disabling functions or modifying the PhotoServe script will earn you a permanent ban.  NO excuses. #####
; ##### If you have found a problem or are looking to make an enhancement please see a PhotoServe author or Channel Operator (in that order) before proceeding #####


/****************************************************************************************
* <b>PSTC - PhotoServe Trigger Configuration module.</b>

* <h4>Conventions</h4><ul>
* <li><b>PSTC_Get*</b> aliases retrieve trigger data. Some of them may also modify the
* data as a side-effect, this is always documented in the alias description.
* <li><b>PSTC_GetGroup*</b> aliases retrieve data about groups. 
* <li><b>PSTC_Set*</b> aliases modify trigger data. They perform no validation of the
* passed values so don't pass junk params. If the second param is omitted (or is $null)
* the corresponding variable is unset.
* <li><b>PSTC_Inc*</b> aliases increment various counters and possibly modify
* other data as well (such as last download/upload date).
* <li><b>PSTC_Configure*</b> aliases prompt user for input and modify trigger data based
* on that input. These aliases do perform validation (check the files actually exist etc).
* </ul>
*
* <b>PsTriggerMap</b> is a lookup table for mapping triggers to groups (*.trg files).
* Back in the olden days pserve would just searh through all trgfiles but that hasn't
* been feasible for many years. The map needs to be rebuilt after creating or removing
* a custom trigger, and after running a trigger update. <code>PSTC_ReloadTrigs</code>
* does that, and a lot more.
* <p>
* <b>PsTriggerArr</b> is an array of the installed triggers. Stored in hash table (the 
* only in-memory data structure mirc has), it maps an integer to a single trigger
* (1 -&gt; SNCD1, 2 -&gt; BGCD5, etc). The array is used for iterating all triggers. We
* could use (and in the past have) <code>$hget(PsTriggerMap,%i).item</code>, but that
* is ~20 times slower and the difference will only grow as more triggers are added.
* <p>
* <b>psCollectionChanged</b> is a signal triggered whenever the state of a trigget changes
* in any way. The signal come with three params: Trig, ChangedSetting, NewValue. For example
* <br><code>on *:SIGNAL:psCollectionChanged: {<br>
*  var %trig = $1, %setting = $2, %newvalue = $3-<br>
*  echo Changed %trig %setting = %newvalue<br></code>
* Might print <tt>"Changed SNCD10 count = 3432"</tt>
* </pre>
* <p>
* <b>Iteration</b> of triggers / groups is done via dedicated iteration aliases, using
* callbacks to run task specific code. Iteration aliases abstract the trigger storage
* format and provide the timer based "anti-freeze" that is needed for most iteration tasks.
* All iteration tasks have a handle (ECHOITER in the examples below) that can be used to
* cancel/pause/resume the task, or to retrieve information about it (see <code>PSTC_Iteration</code>).
* <p>
* Example:<br>
* <code>$PSTC_IterateAllGroups(ECHOITER, async, /echo -s)</code><br>
* <code>$PSTC_IterateGroupTrigs(ECHOITER, sync, SN.TRG, /echo -s)</code><br>
* <p>
* <b>Global variables (settings):</b><ul>
* <li> %ps.lastdirD     -- location of last set collection dir. Used for guessing the location of next set trig.
* <li> %ps.lastdirC     -- like %ps.lastdirD but for CSVs
* <li> %ps.lastdirR     -- like %ps.lastdirD but for reports
* <li> %ps.errorlevel   -- if > 0 leeching is restricted due to too many serious trigger errors
* </ul>
* <b>Transient globals:</b><ul>
* <li> %ps_trigconf_statscache -- Trigger stats are cached for 90 sec to speed up repeat invocations
* </ul>
*/






/****************************************************************************************
* Test whether Trig exists in user's current trigger database.
* @return boolean $true if Trig exists, $false if not
*/
alias PSTC_IsTrig {
  var %trig = $1
  return $iif($PSDB_GetTrig(%trig, name) != $null, $true, $false)
}


/****************************************************************************************
* Returns collection name of Trig. @return string name or $null
*/
alias PSTC_GetName {
  var %trig = $1
  return $PSDB_GetTrig(%trig, name)
}


/****************************************************************************************
* Return current path of Trig (or $null).
* @return path current path of Trig, $null if not set
* @prop valid return the path only if it is valid (exists on disk), else return $null
*/
alias PSTC_GetPath {
  var %trig = $1
  var %path = $PSDB_GetTrig(%trig, path)
  if (%path == $null) return
  var %ps = $regsub(%path, /\\*$/, \\, %path)
  if ($prop == valid) return $iif($isdir(%path), %path)
  return %path
}


/****************************************************************************************
* Return the CSV for trig. Tries to update to newer CSV first (works only with CSVs that
* use the ET naming scheme). Neither <code>GetCSVCount</code> nor <code>GetCSVType</code>
* updates the CSV, so call this alias first if you need to be sure the results for those
* aliases are up to date.<p>
* @return csv full path and filename of the CSV, or $null
* @prop valid return the csv only if it is valid (exists on disk), else return $null
*/
alias PSTC_GetCSV {
  var %trig = $1

  var %current_csv      = $PSDB_GetTrig(%trig, csv)
  var %current_csvfile  = $nopath(%current_csv)
  var %current_csvdir   = $nofile(%current_csv)
  var %current_csvcount = $PSTC_GetCSVCount(%trig)
  var %current_csvstate = $PSTC_GetCsvState(%current_csvfile)

  if ($regex(%current_csvfile, /^(.+?)(\((pre|fin|re).*\))?_\d+\.csv$/i)) {
    var %csv_wildcard = $regml(1) $+ (*)_*.csv $+ ; $+ $regml(1) $+ _*.csv
    var %csv_matchnum = $findfile(%current_csvdir, %csv_wildcard, 0, 1)

    while (%csv_matchnum > 0) {
      var %candidate_csv      = $findfile(%current_csvdir, %csv_wildcard, %csv_matchnum, 1)
      dec %csv_matchnum
      ;; skip the file itself
      if ( %candidate_csv == %current_csv ) continue
      var %candidate_csvfile  = $nopath(%candidate_csv)
      var %candidate_csvstate = $PSTC_GetCsvState(%candidate_csvfile)

      ; Comparing the 2 csvs 
      var %isbetter = $PSL_CsvIsBetter(%current_csvfile,%candidate_csvfile)

      ; If candidate is better no further checking needed
      if (%isbetter) {
        goto updatecsv
      }
      ; Equal CSV state (most likely both are ongoing)
      if (%candidate_csvstate == %current_csvstate) {
        if (!$isfile(%current_csv)) { 
          goto updatecsv
        }
      }

      ; Try next candidate
      continue

      :updatecsv
      if ($PSTC_GetCD(%trig) == $null) {
        echo $ps_n -st --> Updating $+ $ps_kh %trig .csv to $+ $ps_kh %candidate_csvfile $+ $ps_ki -=- Old .csv $+ $ps_kh %current_csvfile
        $PSTC_SetCSV(%trig, %candidate_csv)
        %current_csv      = %candidate_csv
        %current_csvcount = $lines(%candidate_csv)
        %current_csvfile  = %candidate_csvfile
        %current_csvstate = %candidate_csvstate 
      }
      else {
        echo $ps_n -sat --> Found new CSV for $+ $ps_kh %trig but will not automatically update because the trigger has already been burned. 
        echo $ps_n -sat --> Current CSV: $+ $ps_kh %current_csvfile new CSV would have been: $+ $ps_kh %candidate_csvfile
        echo $ps_n -sat --> If the new CSV is correct, please update the CSV setting manually
      }
    }
  }

  if ((%current_csv != $null) && (!$isfile(%current_csv)) && ($PSTC_GetPath(%trig) != $null))  {
    echo $ps_n -sat --> .csv for $+ $ps_kh %trig $+ , %current_csv no longer exists, Please fix this problem.
    if ($window(@PSTriggerError) == $null) {
      PSTC_CheckErrors $me AllChannels $true
    }
  }

  if (($prop == valid) && (!$isfile(%current_csv))) return $null
  return %current_csv
}


/****************************************************************************************
* Return csv state as integer. Higher the number higher the CSV type/revision.
* Only works with CSVs that follow the ET naming scheme.
* @return Integer
*/
alias -l PSTC_GetCsvState {
  var %csvfile = $1
  if ($regex(%csvfile, /^(.+?)(\((pre|fin|re).*?(-\d)?\))?_\d+\.csv$/i)) {
    var %type = $regml(3)
    var %rev  = $abs($regml(4))

    if (%type == pre) return $calc(10 + %rev)
    if (%type == fin) return $calc(20 + %rev)
    if (%type == re)  return $calc(30 + %rev)
  }
  return 0
}


/****************************************************************************************
* Return the type of current CSV. Also works with [most] CSVs that don't follow the ET
* CSV naming scheme. The regexes used here are identical to those in pserve.exe, so
* consistency between the two is guaranteed.
* @return char O for ongoing, P for prefinal, F for final, R for reburn, $null if no CSV.
*/
alias PSTC_GetCSVType {
  var %trig = $1
  var %csvfile = $nopath($PSDB_GetTrig(%trig,csv))
  if (%csvfile == $null) return

  if ($regex(%csvfile, /\(pre-?final.*\)/i)) return P
  if ($regex(%csvfile, /\((re-?burn|revised).*\)/i)) return R
  if ($regex(%csvfile, /[-_(]fin(al|ished)[-_)]?/i)) return F
  return O
}


/****************************************************************************************
* Return CSV-Count for Trig. @return int csv count, $null if no csv has ever been set
* for this trigger
*/
alias PSTC_GetCSVCount {
  var %trig = $1
  return $PSDB_GetTrig(%trig, csvcount)
}


/****************************************************************************************
* Return CRCR32 of CSV for Trig. @return hex csv crc, $null if no csv is set
* for this trigger
*/
alias PSTC_GetCSVCRC {
  var %trig = $1
  var %csv = $PSDB_GetTrig(%trig, csv)
  if ($isfile(%csv)) return $crc(%csv, 2)
}


/****************************************************************************************
* Return the current report for Trig. 
* @return report full path and filename of the report, or $null
* @prop valid return the report only if it is valid (exists on disk), else return $null
*/
alias PSTC_GetReport {
  var %trig = $1
  var %report = $PSDB_GetTrig(%trig, rep)
  if ($prop == valid) return $iif($isfile(%report), %report)
  return %report
}


/****************************************************************************************
* Return current file count for Trig. @return int count or $null
*/
alias PSTC_GetCount {
  var %trig = $1
  return $PSDB_GetTrig(%trig, count)
}


/****************************************************************************************
* Return current byte count for Trig. If path is set for the trigger but size is not,
* this function will perform a count and set the value. @return int size in bytes, $null
* if the size has not been set and the trigger if OFF / not configured.
*/
alias PSTC_GetSize {
  var %trig = $1
  var %size = $PSDB_GetTrig(%trig, size)
  if ((%size == $null) && ($PSTC_GetStatus(%trig).check == ON)) {
    PSTC_Count %trig
    %size = $PSDB_GetTrig(%trig, size)
  }
  return %size
}


/****************************************************************************************
* Return the value of the <code>lastdl</code> variable. @return ctime date and time
*/
alias PSTC_GetLastDL {
  var %trig = $1
  return $PSDB_GetTrig(%trig, lastdl)
}


/****************************************************************************************
* Return the number of days Trig has to remain online (10 day rule).
* @return int
*/
alias PSTC_GetMandatoryOnlineTime {
  var %trig = $1
  var %lastdl = $PSDB_GetTrig(%trig, lastdl)

  var %days = $calc((%lastdl + 10*24*60*60 - $ctime) / (24*60*60) + 0.5)
  return $iif(%days <= 0, 0, $round(%days,0))
}


/****************************************************************************************
* Return current send count for Trig. @return int count or $null
*/
alias PSTC_GetSendCount {
  var %trig = $1
  return $PSDB_GetTrig(%trig, sends)
}


/****************************************************************************************
* Return current status of Trig. @return status ON / OFF, $null for unconfigured trigs
* @prop check check online trig's path for validity, return FAIL status if invalid
*/
alias PSTC_GetStatus {
  var %trig = $1
  var %path = $PSTC_GetPath(%trig)
  if (%path == $null) return

  var %status = $iif($PSDB_GetTrig(%trig, status) == OFF,OFF,ON)
  if (($prop == check) && (%status == ON) && (!$isdir(%path))) return FAIL
  return %status
}


/****************************************************************************************
* Return current CD designation of Trig. @return cdname current CD designation or $null
*/
alias PSTC_GetCD {
  var %trig = $1
  return $PSDB_GetTrig(%trig, cd)
}


/****************************************************************************************
* Return a space separated list of trigs that are on the same CD as Trig. This alias is
* pretty slow (iterates the whole .trg file) so don't call it if you don't have to.
* @return tokens list of triggers, eg: "SNCD1 SNCD2 SNCD3" (without the quotes)
*/
alias PSTC_GetCDTrigs {
  var %trig = $1
  var %cd = $PSTC_GetCD(%trig)
  if (%cd == $null) return

  var %trg = $PSTC_GetTRG(%trig)
  var %cdtrigs
  var %i = $ini(%trg,0)

  while (%i > 0) {
    if ($PSTC_GetCD($ini(%trg, %i)) == %cd) %cdtrigs = %cdtrigs $ini(%trg, %i)
    dec %i
  }
  return $sorttok(%cdtrigs, 32, a)
}


/****************************************************************************************
* Return the URL setting of Trig. In basic form this returns what ever is contents of the
* URL-setting, without any checks for validity. You can enable validity checks with the
* <code>.valid</code> property. The check is just an educated guess though, it cannot
* give an absolute guarantee that the returned URL is valid (let alone that the site 
* still exist). The property will also add the <i>http://</i>-prefix if was missing
* from the URL in .trg file.
* @return string the URL, $null, or possibly something completely different like N/A
* @prop valid do basic syntactic validation of the url, return $null if it fails the check
*/
alias PSTC_GetURL {
  var %trig = $1
  var %url = $PSDB_GetTrig(%trig, url)

  if ($prop == valid) {
    ; Spaces or lack of dot (.) is a sure sign the URL is not valid:
    if ((%url == $null) || ($pos(%url, $chr(32), 1) != $null) || ($pos(%url, ., 1) == $null)) return

    if (http://* !iswm %url) %url = http:// $+ %url
  }
  return %url
}


/****************************************************************************************
* Return the type of Trig. @return type official, 3rdparty, private, custom
*/
alias PSTC_GetType {
  var %trig = $1
  var %group = $PSTC_GetGroup(%trig)
  if (%group == $null) return
  return $PSTC_GetGroupType(%group)
}


/****************************************************************************************
* Return the number of downloaded samples for Trig. If last sample download was more than
* 90 days ago, the counter resets back to zero, allowing users to "refresh their memory",
* and ops to test send ability. @return int number of received samples
*/
alias PSTC_GetSampleCount {
  var %trig = $1

  var %samplestr = $PSDB_GetTrig(%trig, samples)
  if (%samplestr == $null) %samplestr = $readini($PS_SettingsDir $+ PserveSamples.ini", n, %trig, samples)
  if (%samplestr == $null) return 0

  var %gets = $gettok(%samplestr, 1, 59), %time = $gettok(%samplestr, 2, 59)
  if (%time < $calc($ctime - 60*60*24*90)) return 0
  return %gets
}


/****************************************************************************************
* Return time of last file upload of Trig. @return ctime date and time
*/
alias PSTC_GetLastUL {
  var %trig = $1
  return $PSDB_GetTrig(%trig, lastsend)
}


/****************************************************************************************
* Return Trig's current BitTorrent status.
* @return BT-status 'BTS' if Trig is being seeded, 'BTL' if leeched, $null if neither
*/
alias PSTC_GetBTState {
  var %trig = $1
  var %btstate = $PSDB_GetTrig(%trig, btstate)
  if (%btstate == $null) return 
  if ($PSTC_GetStatus(%trig) != ON) return

  if (%btstate == locked) return BTL
  if (%btstate == lock) return BTL
  if (%btstate == seed) return BTS
}


/****************************************************************************************
* Return the .trg file (with full path) in which Trig resides. Usually you should use
* the the PSTC-API to read and write .trg files, but in some cases accessing the files
* directly via readini/writeini is justified (usually for speed). The returned filename
* is not double quoted.
* @return trg trigger file with full path, or $null
*/
alias PSTC_GetTRG {
  var %trig = $1
  if ($hget(PsTriggerMap, %trig) == $null) return $null
  return $remove($PS_TriggersDir $+ $hget(PsTriggerMap, %trig), ")
}


/****************************************************************************************
* Return the group in which Trig resides. With the current .trg storage format
* <code>group</code> is the same as the group filename without path (eg SN.TRG).
* @return group 
*/
alias PSTC_GetGroup {
  var %trig = $1
  return $hget(PsTriggerMap, %trig)
}


/****************************************************************************************
* Return the group of the specified group name. Not the fastest function so use sparingly.
* @return group 
*/
alias PSTC_GetGroupByName {
  var %groupname = $1

  var %triggersdir = $PS_TriggersDir
  var %group
  var %ps = $findfile(%triggersdir, *.trg, 0, 1, if ($readini($1-,n,info,name) == %groupname) %group = $nopath($1-))
  return %group
}


/****************************************************************************************
* Return the type of Group. If there is no type defined in the .trg file, sets it to
* custom. If Group does not exist, returns $null.
* @return type official, 3rdparty, private, custom or $null
*/
alias PSTC_GetGroupType {
  var %group = $1

  if (!$isfile($PS_TriggersDir $+ %group)) return

  var %type = $PSDB_GetGroup(%group, type)
  if (%type != $null) return %type
  $PSDB_SetGroup(%group, type, custom)
  return custom
}


/****************************************************************************************
* Return the name of Group. @return string name or $null
*/
alias PSTC_GetGroupName {
  var %group = $1
  var %folder = $PSDB_GetGroup(%group, subfolder)
  if (%folder == $null) return $PSTC_GetGroupShortname(%group)
  return %folder - $PSTC_GetGroupShortname(%group)
}


/****************************************************************************************
* Return Groups's subfolder setting ($null if no subfolder)
* @return subfolder the subfolder setting from group's [INFO] section, or $null
*/
alias PSTC_GetGroupSubfolder {
  var %group = $1
  return $PSDB_GetGroup(%group, subfolder)
}


/****************************************************************************************
* Return Groups's shortname setting. If shortname is not set, sets shortname to same
* name and returns that. If neither is set (should never happen) returns $null
* @return shortname the shortname setting from group's [INFO] section
*/
alias PSTC_GetGroupShortname {
  var %group = $1
  var %sname = $PSDB_GetGroup(%group, shortname)
  if (%sname == $null) {
    %sname = $PSDB_GetGroup(%group, name)  
    $PSDB_SetGroup(%group, shortname, %sname)  
  }  
  return %sname
}




/****************************************************************************************
* Set new Path for Trig. @optparam path @return void
*/
alias PSTC_SetPath {
  var %trig = $1, %path = $2-

  if (%path == $null) {
    $PSDB_SetTrig(%trig, path)
  }
  else {
    var %ps = $regsub($remove(%path,"), /\\*$/, \\, %path)
    $PSDB_SetTrig(%trig, path, %path)
  }
}

/****************************************************************************************
* Set new CSV and update CSV count for Trig. @optparam csv @return void
*/
alias PSTC_SetCSV {
  var %trig = $1, %csv = $2-
  $PSDB_SetTrig(%trig, csv, %csv)
  if ($regex($nopath(%csv), /_(\d+)\.csv"?$/i)) $PSDB_SetTrig(%trig, csvcount, $regml(1))
  elseif ($isfile(%csv))                        $PSDB_SetTrig(%trig, csvcount, $lines(%csv))
  if (($dialog(PSTC_ConfigDlg) != $null) && (%ps_trigconf_dlgtrig == %trig)) did -ra PSTC_ConfigDlg 20 %csv
}

/****************************************************************************************
* Set new Report for Trig. @optparam report @return void
*/
alias PSTC_SetReport {
  var %trig = $1, %report = $2
  $PSDB_SetTrig(%trig, rep, %report)
}

/****************************************************************************************
* Set last DL time (in seconds) for Trig. @return void
*/
alias PSTC_SetLastDL {
  var %trig = $1 , %ctime = $2
  $PSDB_SetTrig(%trig, lastdl, %ctime)
}

/****************************************************************************************
* Set Trig to Status. If user is about to turn trigger off and last download was less
* than 10 days ago, a confirmation box is shown. If the trigger is on CD/DVD and there
* are other triggers on the CD, ask whether the change should apply to them also.
* @return void
*/
alias PSTC_SetStatus {
  var %trig = $1, %status = $2
  if (($PSTC_GetMandatoryOnlineTime(%trig) > 0) && (%status == OFF)) {
    if (!$?!="Turning %trig off would violate the 10-Day-Rule. $crlf $+ Are you sure you want to proceed and turn %trig off?") return
  }

  if (($disk($PSTC_GetPath(%trig)).type == cdrom) && ($PSTC_GetCD(%trig) != $null)) {
    var %cdtrigs = $PSTC_GetCDTrigs(%trig)
    var %changes = 0

    var %i = $numtok(%cdtrigs, 32)
    while (%i > 0) {
      if ($PSTC_GetStatus($gettok(%cdtrigs, %i, 32)) != %status) inc %changes
      dec %i
    }

    if ((%changes > 1) && ($?!="Turn %status all triggers on this CD/DVD? $crlf $+ ( $+ %cdtrigs $+ )")) {
      var %i = $numtok(%cdtrigs,32)
      while (%i > 0) {
        $PSDB_SetTrig($gettok(%cdtrigs, %i, 32), status, %status)
        dec %i
      }
      return
    }
  }
  $PSDB_SetTrig(%trig, status, %status)
}

/****************************************************************************************
* Set Trig's current BitTorrent status.
* @param state either BTL or BTS or anything including $null (to clear)
* @return void
*/
alias PSTC_SetBTState {
  var %trig = $1 , %btstate = $2
  if (%btstate == BTL) var %state = lock
  elseif (%btstate == BTS) var %state = seed
  else var %state = $null
  $PSDB_SetTrig(%trig, btstate, %state)
}


/****************************************************************************************
* Add Trig to CD / change CD designation.
* @optparam cd @return void
*/
alias PSTC_SetCD {
  var %trig = $1, %cd = $2
  $PSDB_SetTrig(%trig, CD, %cd)
}

/****************************************************************************************
* Increment file count of Trig by one and update its <code>lastdl</code> value to
* current date and time. @return void
*/
alias PSTC_IncFileCount {
  var %trig = $1
  var %count = $PSTC_GetCount(%trig)
  inc %count
  $PSDB_SetTrig(%trig, count, %count)
  $PSDB_SetTrig(%trig, lastdl, $ctime)
}

/****************************************************************************************
* Increment byte count (size) of Trig by Addbytes. @return void
*/
alias PSTC_IncByteCount {
  var %trig = $1, %addbytes = $2

  var %size = $PSTC_GetSize(%trig) + %addbytes
  $PSDB_SetTrig(%trig, size, %size)
}

/****************************************************************************************
* Increment Trig send count by one and update its <code>lastsend</code> value to
* current date and time. @return void
*/
alias PSTC_IncSendCount {
  var %trig = $1 , %bytes = $2
  var %sends = $PSTC_GetSendCount(%trig)
  inc %sends
  $PSDB_SetTrig(%trig, sends, %sends)
  $PSDB_SetTrig(%trig, lastsend, $ctime)
  if (%bytes > 0) {
    var %sentbytes = $iif($PSTC_GetRawVar(%trig,sentbytes),$ifmatch,0)
    inc %sentbytes %bytes
    $PSDB_SetTrig(%trig, sentbytes, %sentbytes)
  }
}

/****************************************************************************************
* Increment sample counter by one and update the last sample date. @return void
*/
alias PSTC_IncSampleCount {
  var %trig = $1

  var %gets = 1 + $PSTC_GetSampleCount(%trig)
  if ($PSTC_GetGroup(%trig) != $null) $PSDB_SetTrig(%trig, samples, %gets $+ ; $+ $ctime)
  else writeini -n $PS_SettingsDir(PSTC_IncSampleCount) $+ PserveSamples.ini" %trig samples %gets $+ ; $+ $ctime
}



/****************************************************************************************
* Opens modal file selection dialog allowing user to set Path. @return void
*/
alias PSTC_ConfigurePath {
  var %trig = $1
  var %path = $PSTC_SelectPath(%trig)
  if (%path == $null) return
  $PSTC_SetPath(%trig, %path)
  $PSTC_Count(%trig)
}

/****************************************************************************************
* Asks user to select path and returns the selection. @return path selected path or $null
*/
alias -l PSTC_SelectPath {
  var %trig = $1

  ; Try previus trigger in series
  if ($regex(%trig,/^(.+?)(\d+)$/)) {
    var %prevtrig = $regml(1) $+ $calc($regml(2) - 1)
    var %searchpath = $nofile($left($PSTC_GetPath(%prevtrig), -1))
  }
  ; Try last path used in this group
  if (!$isdir(%searchpath)) {
    var %grouppath = $PSDB_GetGroup($PSTC_GetGroup(%trig), lastpath)
    var %searchpath = $nofile($left(%grouppath, -1))
  }
  ; Try last path used in ANY group
  if (!$isdir(%searchpath)) {
    var %searchpath = %ps.lastdirD
  }
  ; No history data available, fall back to mircdir
  if (!$isdir(%searchpath)) {
    var %searchpath = $mircdir
  }

  var %path = $sdir(%searchpath, "Select Path")

  if ($isdir(%path)) {
    set %ps.lastdirD $nofile($left(%path, -1)) 
    $PSDB_SetGroup($PSTC_GetGroup(%trig), lastpath, %path)
    return %path
  }
}



/****************************************************************************************
* Opens modal file selection dialog allowing user to set CSV. @return void
*/
alias PSTC_ConfigureCSV {
  var %trig = $1
  var %csv = $PSTC_SelectCSV(%trig)
  if (%csv == $null) return
  $PSTC_SetCSV(%trig, %csv)
}

/****************************************************************************************
* Asks user to select CSV and returns the selection.
* @return csv full path and filename of the CSV, or $null
*/
alias -l PSTC_SelectCSV {
  var %trig = $1

  ; Try previus trigger in series
  if ($regex(%trig,/^(.+?)(\d+)$/)) {
    var %prevtrig = $regml(1) $+ $calc($regml(2) - 1)
    var %searchpath = $nofile($left($PSTC_GetCSV(%prevtrig), -1))
  }
  ; Try last path used in this group
  if (!$isdir(%searchpath)) {
    var %groupcsv = $PSDB_GetGroup($PSTC_GetGroup(%trig), lastcsv)
    var %searchpath = $nofile($left(%groupcsv, -1))
  }
  ; Try last path used in ANY group
  if (!$isdir(%searchpath)) {
    var %searchpath = %ps.lastdirC
  }
  ; No history data available, fall back to mircdir
  if (!$isdir(%searchpath)) {
    var %searchpath = $mircdir
  }

  var %csv = $sfile(%searchpath $+ *.csv, "Select CSV")

  if ($isfile(%csv)) {
    set %ps.lastdirC $nofile(%csv)
    $PSDB_SetGroup($PSTC_GetGroup(%trig), lastcsv, %csv)
    return %csv
  }
}



/****************************************************************************************
* Opens modal file selection dialog allowing user to set report. If a report is
* selected, tries to extract Path and CSV info from it as well. @return void
*/
alias PSTC_ConfigureReport {
  var %trig = $1

  var %report = $PSTC_SelectReport(%trig)
  if (%report != $null) {
    var %csv = $PSTC_ExtractCSVFromReport(%report)
    var %path = $PSTC_ExtractPathFromReport(%report)
  }

  if (%report != $null) $PSTC_SetReport(%trig, %report)
  if (($PSTC_GetCSV(%trig) == $null) && (%csv != $null)) $PSTC_SetCSV(%trig, %csv)
  if (($PSTC_GetPath(%trig) == $null) && (%path != $null)) {
    $PSTC_SetPath(%trig, %path)
    $PSTC_Count(%trig)
  }
}

/****************************************************************************************
* Asks user to select Report and returns the selection.
* @return report full path and filename of the report, or $null
*/
alias -l PSTC_SelectReport {
  var %trig = $1

  ; Try previus trigger in series
  if ($regex(%trig,/^(.+?)(\d+)$/)) {
    var %prevtrig = $regml(1) $+ $calc($regml(2) - 1)
    var %searchpath = $nofile($left($PSTC_GetReport(%prevtrig), -1))
  }
  ; Try last path used in this group
  if (!$isdir(%searchpath)) {
    var %groupcsv = $PSDB_GetGroup($PSTC_GetGroup(%trig), lastreport)
    var %searchpath = $nofile($left(%groupcsv, -1))
  }
  ; Try last path used in ANY group
  if (!$isdir(%searchpath)) {
    var %searchpath = %ps.lastdirR
  }
  ; No history data available, fall back to mircdir
  if (!$isdir(%searchpath)) {
    var %searchpath = $mircdir
  }

  var %rep = $sfile(%searchpath $+ *.txt, "Select Report")

  if ($isfile(%rep)) {
    set %PS.lastdirR $nofile(%rep)
    $PSDB_SetGroup($PSTC_GetGroup(%trig), lastreport, %rep)
    return %rep
  }
}



/****************************************************************************************
* Attempt to read Report to determine the collection path. 
* @return path the extracted path, or $null
*/
alias -l PSTC_ExtractPathFromReport {
  var %report = $1-

  if ($remove($read(%report, rn, /^In Directory:\s*(.+?)\s*$/i), ") != $null) var %path = $regml(1)
  elseif ($remove($read(%report, rn, /^Collection:\s*(.+?)\s*$/i), ") != $null) var %path = $regml(1)

  if (!$isdir(%path)) return
  var %ps = $regsub(%path, /\\*$/, \\, %path)
  return %path
}



/****************************************************************************************
* Attempt to read Report to determine the collection CSV. PicCheck and ScanSort only
* save the filename of the CSV, not the the path. In those cases we do a search the
* CSV-dir for matching filename. Seriously, this thing supports ScanSort.
* @return csv the extracted csv, or $null
*/
alias -l PSTC_ExtractCSVFromReport {
  var %report = $1-

  if ($remove($read(%report, rn, /^CSV file:\s*(.+\.csv)/i ), ") != $null)                         var %csv = $regml(1)
  elseif ($remove($read(%report, rn, /^List Source: \([A-F0-9]{1,8}\) (.+\.csv)/i ), ") != $null)  var %csv = $regml(1)
  elseif ($remove($read(%report, rn, /^List Source: (.+\.csv)/i ), ") != $null)                    var %csv = $regml(1)

  if (\ isin %csv) {
    if ($isfile(%csv)) return %csv
  }
  else {
    return $findfile(%ps.csvdir, %csv, 1, 2)
  }
}



/****************************************************************************************
* Open modeless path/csv/report configuration dialog for Trig.
* TODO: Update dialog when path/csv/rep are changed from elsewhere (like ET-dialog).
* @return void
*/
alias PSTC_ConfigureAll {
  var %trig = $1

  set %ps_trigconf_dlgtrig %trig
  if ($dialog(PSTC_ConfigDlg) != $null) dialog -x PSTC_ConfigDlg
  dialog -m $+ %PS.dlg.ontop $+ $PSA_GetDesktopFlag(dialogs) PSTC_ConfigDlg PSTC_ConfigDlg
}

dialog PSTC_ConfigDlg {
  title Setup %ps_trigconf_dlgtrig
  size -1 -1 270 75
  option dbu

  box "Setup Trigger" 10, 5 3 260 71
  text "Path" 12, 10 14 15 10
  edit "" 14, 30 13 170 10, autohs
  button "Browse" 16, 235 13 25 10
  button "Create" 50, 205 13 25 10
  text "CSV" 18, 10 26 15 10
  edit "" 20, 30 25 170 10, autohs read
  button "Browse" 22, 235 25 25 10
  button "Lookup" 51, 205 25 25 10

  box "Set report first, in most cases photoserve can extract Path and/or CSV info from it" 38, 7 36 226 22
  text "Report" 24, 10 45 16 10
  edit "" 26, 30 44 170 10, autohs read
  button "Browse" 28, 205 44 25 10

  text "Configure Directories/Files for each Line and select OK" 30, 10 61 165 10
  button  "OK" 32, 175 60 25 10, ok disable
  button  "Clear" 34, 205 60 25 10
  button  "Cancel" 36, 235 60 25 10, cancel default
}

on *:dialog:PSTC_ConfigDlg:init:0: {
  var %trig = %ps_trigconf_dlgtrig

  did -ra $dname 14 $PSTC_GetPath(%trig)
  did -ra $dname 20 $PSTC_GetCSV(%trig).valid
  did -ra $dname 26 $PSTC_GetReport(%trig).valid

  if (($did($dname, 14) != $null) && ($did($dname, 20) != $null)) did -e $dname 32
}

on *:dialog:PSTC_ConfigDlg:sclick:*: {
  var %trig = %ps_trigconf_dlgtrig

  if ($did == 16) {
    var %path = $PSTC_SelectPath(%trig)
    if (%path != $null) did -ra $dname 14 %path
  }
  if ($did == 22) {
    var %csv = $PSTC_SelectCSV(%trig)
    if (%csv != $null) did -ram $dname 20 %csv
  }
  if ($did == 28) {
    var %report = $PSTC_SelectReport(%trig)
    if (%report != $null) {
      did -ram $dname 26 %report

      var %csv = $PSTC_ExtractCSVFromReport(%report)
      if (%csv != $null) did -ram $dname 20 %csv

      var %path = $PSTC_ExtractPathFromReport(%report)
      if (%path != $null) did -ram $dname 14 %path
    }
  }
  if (($did($dname, 14) != $null) && ($did($dname, 20) != $null)) did -e $dname 32

  ; OK
  if ($did == 32) {
    $PSTC_SetPath(%trig, $did($dname, 14).text)
    $PSTC_SetCSV(%trig, $did($dname, 20).text)
    $PSTC_SetReport(%trig, $did($dname, 26).text)
    $PSTC_Count(%trig)
  }
  ; Clear
  if ($did == 34) {
    did -rm $dname 14,20,26
    did -b $dname 32
  }
  ; Create Dir
  if ($did == 50) {
    did -m $dname 14
    did -e $dname 32
    var %ps_mp = $PS_MakePath($did($dname, 14).text)
  }
  ; CSV Lookup
  if ($did == 51) {
    PSCSV_LookupTrig %trig
  }
}



/****************************************************************************************
* Modal dialog for setting trigger backup ID. @return void
*/
alias PSTC_ConfigureCD {
  var %trig = $1 

  var %oldcd = $PSTC_GetCD(%trig)
  var %newcd = $?="Enter backup ID (current ID: %oldcd $+ )"

  if ((%newcd == $null) && (%oldcd != $null)) {
    if (!$?!="Do you want to remove current backup ID setting ( $+ %oldcd $+ )") return
  }

  $PSDB_SetTrig(%trig, cd, $upper(%newcd))
}



/****************************************************************************************
* Runs a full count on trig and updates trigger DB. Offline and unconfigure triggers
* silently ignored, making it posssible to call this via the iteration aliases.
* <p>
* The counting is performed by DLL due mirc being too slow at it. Mirc can count with
* reasonable speed but getting the filesize is very slow. PSTC_Count_mircscript is a
* mircscript-only counting function, but it has not been updated to count filesizes.
* @return void
*/
alias PSTC_Count {
  var %trig = $1

  var %path = $PSTC_GetPath(%trig).valid
  if (%path == $null) return
  if ($PSTC_GetStatus(%trig) != ON) return
  echo $ps_n -st --> Counting: $+ $ps_kh %trig

  ; Result = 367832525 bytes in 324 files
  var %result = $dll($PS_ProgramDir $+ pserve.dll", Count, " $+ %path $+ " )

  if (* bytes in * files iswm %result) {
    var %size = $gettok(%result, 1, 32)
    var %count = $gettok(%result, 4, 32)
    var %oldcount = $PSTC_GetCount(%trig)
    $PSDB_SetTrig(%trig, count, %count)
    $PSDB_SetTrig(%trig, size, %size)
    if (%count != %oldcount) {
      echo $ps_n -sat --> Count for $+ $ps_kh %trig updated to $+ $ps_kh %count $+ , old count $+ $ps_kh $iif(%oldcount != $null, %oldcount, -)
    }
    if (%count > $PSTC_GetCSVCount(%trig)) {
      echo $ps_n -sat --> 4ERROR Collection $+ $ps_kh %trig Path $+ $ps_kh %path Contains BAD or INCORRECT Files.
      echo $ps_n -sat --> 4ERROR You may have a folder named $+ $ps_kh \bad\ or $+ $ps_kh \unknown\ which contains incorrect files OR you have the $+ $ps_kh .csv file in the triggers path (4not allowed) or $+ $ps_kh descript.ion or $+ $ps_kh thumbs.db or $+ $ps_kh desktop.ini is somewhere in the triggers folders and is causing a bad count.  03DELETE THEM!
    }
  }
}


/****************************************************************************************
* Runs a full count on trig and updates trigger DB. Offline and unconfigure triggers
* silently ignored, making it posssible to call this via the iteration aliases
* @return void
*/
alias PSTC_Count_mircscript {
  var %trig = $1

  var %path = $PSTC_GetPath(%trig).valid
  if (%path == $null) return
  if ($PSTC_GetStatus(%trig) != ON) return

  echo $ps_n -sat --> Counting: $+ $ps_kh %trig
  set %ps_trigconf_count_badfiles 0
  var %count = $findfile(%path, *, 0, PSTC_CountBadFiles $len(%path) $1-)

  if (%ps_trigconf_count_badfiles > 0) {
    echo $ps_n -sat --> 4ERROR Collection $+ $ps_kh %trig Path $+ $ps_kh %path Contains04 %ps_trigconf_count_badfiles BAD or INCORRECT Files.
    echo $ps_n -sat --> 4ERROR You may have a folder named $+ $ps_kh \bad\ or $+ $ps_kh \unknown\ which contains incorrect files OR you have the $+ $ps_kh .csv file in the triggers path (4not allowed) or $+ $ps_kh descript.ion or $+ $ps_kh thumbs.db or $+ $ps_kh desktop.ini is somewhere in the triggers folders and is causing a bad count.  03DELETE THEM!
    %count = %count - %ps_trigconf_count_badfiles
  }
  unset %ps_trigconf_count_badfiles

  var %oldcount = $PSTC_GetCount(%trig)
  if (%count != %oldcount) echo $ps_n -sat --> Count for $+ $ps_kh %trig updated to $+ $ps_kh %count $+ , old count $+ $ps_kh $iif(%oldcount != $null, %oldcount, -)
  $PSDB_SetTrig(%trig, count, $calc(%count - %ps_trigconf_count_badfiles))
}


/****************************************************************************************
* Count number of bad files (thumbnail, csv, etc) during count.
* @param trigpathlen lenght of the trigger path
* @return void
*/
alias -l PSTC_CountBadFiles {
  var %trigpathlen = $1, %pathfile = $2-
  tokenize 92 $right(%pathfile, - $+ %trigpathlen)
  if (($1 == bad) || ($1 == unknown) || (descript.ion isin $1-) || (thumbs.db isin $1-) || (desktop.ini isin $1-) || (.csv isin $1-)) inc %ps_trigconf_count_badfiles
}



/****************************************************************************************
* Unsets Trig, after prompting the user for confirmation if Trig is over the preview
* limit. Preview is defined as 300 files / 10 vids / 160 MB (exceeding any one of those
* three means it is over the preview). Custom triggers can always be unset. @return void
*/
alias PSTC_UnsetTrig {
  var %trig = $1

  if ($PSTC_GetType(%trig) == custom) goto unset
  if ($PSTC_GetCount(%trig) > 300) goto confirm

  ; Vid and size checks are useless if the files have been deleted from the dir, which is probably the case.. :/
  var %vidcount = $calc( $findfile($PSTC_GetPath(%trig), *.mov, 0) + $findfile($PSTC_GetPath(%trig), *.mpg, 0) + $findfile($PSTC_GetPath(%trig), *.mpeg, 0) + $findfile($PSTC_GetPath(%trig), *.wmv, 0) + $findfile($PSTC_GetPath(%trig), *.avi, 0) )
  if (%vidcount > 10) goto confirm

  set %ps_trigconf_unset_size 0
  var %ps = $findfile($PSTC_GetPath(%trig), *.*, 0, set %ps_trigconf_unset_size $calc($file($1-).size + %ps_trigconf_unset_size))
  var %mbsize = $round($calc(%ps_trigconf_unset_size / 1024 / 1024), 1)
  if (%mbsize > 160) goto confirm

  goto unset

  :confirm
  var %warning = PhotoServe thinks unsetting $upper(%trig) is against the rules (trigger is over the preview limit). $&
    In some cases channel ops can give you permission to unset triggers anyway, unsetting without permission $&
    will likely get you banned. $+ $crlf $+ $crlf $+ Are you ABSOLUTELY SURE you want to unset this trigger?
  if ($input(%warning, yw, Unset Trigger?)) goto unset
  return

  :unset
  $PSDB_SetTrig(%trig, path)
  $PSDB_SetTrig(%trig, status)
  $PSDB_SetTrig(%trig, count)
  $PSDB_SetTrig(%trig, csv)
  $PSDB_SetTrig(%trig, csvcount)
  $PSDB_SetTrig(%trig, rep)
  $PSDB_SetTrig(%trig, have)
  $PSDB_SetTrig(%trig, cd)
  $PSDB_SetTrig(%trig, sends)
  $PSDB_SetTrig(%trig, lastdl)
  $PSDB_SetTrig(%trig, lastsend)
  $PSDB_SetTrig(%trig, btstate)
}



/****************************************************************************************
* Present user with a modeless dialog for creating new custom trigger. You can prefill
* the form by specifying Group, Trig and Name, but user has last say on all of these. 
* Since this lovely programming language doesn't support $null params, specify some
* nonsense BS like <code>N/A</code> for Group if you want to specify Trig and Name
* without preselecting group. Such invalid group is silently ignored, the combobox just
* stays unfilled.
* @optparam group @optparam trig @optparam name @return void
*/
alias PSTC_AddCustomTrig {
  var %group = $1, %trig = $2, %name = $3-
  set %ps_trigconf_newtrig_trig %trig
  set %ps_trigconf_newtrig_group %group
  set %ps_trigconf_newtrig_name %name

  if ($dialog(PSTC_NewTrigDlg) != $null) dialog -x PSTC_NewTrigDlg
  dialog -m $+ %PS.dlg.ontop $+ $PSA_GetDesktopFlag(dialogs) PSTC_NewTrigDlg PSTC_NewTrigDlg
}

dialog PSTC_NewTrigDlg {
  title "Add New Custom Collection"
  size -1 -1 432 164

  box "Create Trigger" 1, 10 8 412 116

  text "In Group" 2, 20 32 42 16
  combo 3, 70 30 230 130, drop sort
  Button "Add New Group" 4, 308 30 100 22, read

  text "Name" 5, 20 62 28 16
  edit "" 6, 69 60 340 22, autohs

  text "Trigger" 7, 20 92 36 20
  edit "" 8, 69 90 86 22, autohs

  text "" 10, 10 136 300 20
  button "Add" 11, 316 134 51 22, ok disable
  button "Cancel" 12, 373 134 49 22, cancel default
}

;; Prefill fields with values passed to PSTC_AddCustomTrig
on *:dialog:PSTC_NewTrigDlg:init:0: {
  PSTC_IterateAllGroups PSTC_CustTrig sync PSTC_NewTrigDlg_InitGroups 
  did -a $dname 6 %ps_trigconf_newtrig_name
  did -a $dname 8 %ps_trigconf_newtrig_trig
  unset %ps_trigconf_newtrig_*
  .timer 1 0 PSTC_NewTrigDlg_Validate
}

/****************************************************************************************
* Adds custom Group to NewTrigDlg's combobox. Called via <code>PSTC_IterateAllGroups</code>
* @return void
*/
alias -l PSTC_NewTrigDlg_InitGroups {
  var %group = $1
  if ($PSTC_GetGroupType(%group) == custom) {
    did -a $+ $iif(%ps_trigconf_newtrig_group == %group,c) PSTC_NewTrigDlg 3 $PSTC_GetGroupName(%group) : %group
  }
}

;; Active validation for Trig and Name of new custom collection
on *:dialog:PSTC_NewTrigDlg:edit:*: {
  PSTC_NewTrigDlg_Validate
}

/****************************************************************************************
* Validates the PSTC_NewTrigDlg form. Makes sure name and ID only contain allowed chars
* and that a group has been selected from the combobox. If the form is valid enables the
* Add-button, else disables it and specifies the error. @param void @return void
*/
alias -l PSTC_NewTrigDlg_Validate {
  var %dname = PSTC_NewTrigDlg
  if ($dialog(%dname) == $null) return

  var %name = $did(%dname,6).text
  var %trig = $upper($did(%dname,8).text)
  var %group = $gettok($did(%dname,3).text, -1, $asc(:))

  if ($regex(%trig, /[^A-Z0-9-]/)) {
    did -ra %dname 10 Trigger should contain only letters A-Z, numbers 0-9, and -
    did -b %dname 11
  }
  elseif ($PSTC_GetGroup(%trig) != $null) {
    did -ra %dname 10 %trig already exists ( $+ $PSTC_GetName(%trig) $+ )
    did -b %dname 11
  }
  elseif ($regex(%name, /[^\w\#\&\-\(\) ]/)) {
    did -ra %dname 10 Name should contain only these characters: A-Z 0-9 -_&&#( )
    did -b %dname 11
  }
  elseif ((%trig != $null) && (%name != $null) && (%group != $null)) {
    did -ra %dname 10 OK... This trigger is new and formatted correctly
    did -e %dname 11
  }
  else {
    did -r %dname 10
    did -b %dname 11
  }
}

;; Selected group from combobox, revalidate form (enables Add button)
on *:dialog:PSTC_NewTrigDlg:sclick:3: {
  PSTC_NewTrigDlg_Validate
}

;; "Add New Group"-button
on *:dialog:PSTC_NewTrigDlg:sclick:4: {
  PSTC_AddCustomGroup
  did -r $dname 3
  $PSTC_IterateAllGroups(PSTC_CustTrig, sync, PSTC_NewTrigDlg_InitGroups)
  dialog -v $dname
}

;; "Add"-button, creates new custom collection
on *:dialog:PSTC_NewTrigDlg:sclick:11: {
  var %name = $did($dname,6).text, %trig = $upper($did($dname,8).text)
  var %group = $gettok($did($dname,3).text, -1, $asc(:))

  hadd PsTriggerMap %trig %group
  hadd PsTriggerArr $calc($hget(PsTriggerArr,0).item + 1) %trig
  $PSDB_SetTrig(%trig, name, %name)

  .signal PS2PSC_ReloadTriggers
}



/****************************************************************************************
* Presents user with a modal dialog for creating a new custom group. <p>TODO: replace
* prompts with a dialog similar to PSTC_NewTrigDlg. @param void @return void
*/
alias PSTC_AddCustomGroup {
  var %gname = $??="Enter a name for this group"
  if (%gname == $null) return

  var %prompt = Enter an ID for this group
  :askgroup
  var %gid = $??=" %prompt "
  if (%gid == $null) return
  var %group = Cust_ $+ %gid $+ .TRG

  if ($regex(%gid, /\W/)) {
    %prompt = $??="ID should only contain characters A-Z and 0-9"
    goto askgroup
  }
  if ($isfile($PS_TriggersDir $+ %group $+ ")) {
    %prompt = $??="That ID already exists, try again. Enter a new ID for this group"
    goto askgroup
  }

  $PSDB_SetGroup(%group, name, %gname)
  $PSDB_SetGroup(%group, type, custom)

  .signal PS2PSC_ReloadTriggers

  ; Touching trigrev file causes PSCPro to reload triggers
  var %revfile = $PS_TriggersDir(PSTC_AddCustomGroup) $+ PSTrigRev.txt"
  var %oldsize = $file(%revfile).size
  write %revfile Just touching the file..
  btrunc %revfile %oldsize
}



/****************************************************************************************
* Removes Trig, provided it is custom. @return void
*/
alias PSTC_RemoveCustomTrig {
  var %trig = $1

  if ($PSTC_GetType(%trig) != custom) {
    echo $ps_n -sat --> You can only remove custom triggers, $ps_kh $+ $PSTC_GetName(%group) is not custom
  }
  else {
    echo $ps_n -sat --> Removed custom trigger $+ $ps_kh $PSTC_GetName(%trig)
    remini -n $PS_TriggersDir $+ $PSTC_GetGroup(%trig) $+ " %trig
    .timer 1 1 PSTC_ReloadTriggers

    .signal PS2PSC_ReloadTriggers
  }
}



/****************************************************************************************
* Removes Group, provided it is both custom and empty. @return void
*/
alias PSTC_RemoveCustomGroup {
  var %group = $1
  var %trgfile = $PS_TriggersDir(PSTC_RemoveCustomGroup) $+ %group $+ "

  if ($PSTC_GetGroupType(%group) != custom) {
    echo $ps_n -sat --> You can only remove custom groups, $ps_kh $+ $PSTC_GetGroupName(%group) is not custom
  }
  elseif ($ini(%trgfile, 0) > 1) {
    echo $ps_n -sat --> This group $+ $ps_kh $PSTC_GetGroupName(%group) can not be removed. Remove all the triggers first
  }
  else {
    echo $ps_n -sat --> Removed custom group $+ $ps_kh $PSTC_GetGroupName(%group)
    .remove %trgfile

    .signal PS2PSC_ReloadTriggers

    ; Touching trigrev file causes PSCPro to reload triggers
    var %revfile = $PS_TriggersDir(PSTC_RemoveCustomGroup) $+ PSTrigRev.txt"
    var %oldsize = $file(%revfile).size
    write %revfile Just touching the file..
    btrunc %revfile %oldsize
  }
}



/****************************************************************************************
* Runs error check when an op types <tt>!PSChkTrgs</tt> in channel 
*/ 
on ^*:TEXT:!PSChkTrgs*:#: {
  haltdef
  if (($nick isop $chan) && ($PSC_isPSonInChannel($chan))) {
    echo $ps_n -st --> Channel-wide Trigger Check requested by: $+ $ps_kh $nick on: $+ $ps_kh $chan
    PSTC_CheckErrors $nick $chan $true
  }
}

/****************************************************************************************
* Runs error check on private notice of <tt>!PSChkTrgs #Chan</tt>
*/ 
on ^*:NOTICE:!PSChkTrgs*:?: {
  haltdef
  if (($me ison $2) && ($nick ison $2) && ($PSC_isPSonInChannel($2))) {
    echo $ps_n -sat --> Trigger Check requested by: $+ $ps_kh $nick on: $+ $ps_kh $2
    PSTC_CheckErrors $nick $2 $false
  }
}


/****************************************************************************************
* Calls pserve.exe to run a trigger error check. 
* @param nick the person who requested the check (who to send the reply to)
* @param chan channel for which the check was done
* @param chanwide $true if the the check is run for the whole channel, $false for private check
* @return void
*/
alias PSTC_CheckErrors {
  var %nick = $1, %chan = $2, %chanwide = $3

  if ($timer(PSTC_ERRCHK)) {
    if (%nick != $me) .notice %nick Already running !PSChkTrgs for another user - try again in a couple of minutes.
  }
  else {
    if ($isfile($PS_TriggersDir $+ TrigChk.txt")) .remove $PS_TriggersDir $+ TrigChk.txt"
    .timerPSTC_ERRCHK 300 1 PSTC_PollErrorFile %nick %chanwide
    ; Check can take up to 2 minutes so we have to use the non-blocking /run command instead of pserve.dll
    run $PS_ProgramDir $+ pserve.exe" CheckTrigs $left($PS_ScriptDir,-1) $+ " %chan
  }
}

/****************************************************************************************
* Polls triggers dir once per second to see if the trigger error check has been
* completed. Appearance of file TrigChk.txt signals that the check is done. Reads
* the results and send replies.
* <p>Also sets the %ps.errorlevel global variable, which tells leech system that so
* many errors were found that user should not be allowed to leech until they are fixed.
* %ps.errorlevel == 0 means allow leeching, %ps.errorlevel == 1 means block new leech
* windows, %ps.errorlevel = 2 means stop leeching entirely.
* <p>TODO: schedule a recheck if leech-stopping errors are found (60 minute timer).
* @return void
*/
alias -l PSTC_PollErrorFile {
  var %nick = $1, %chanwide = $2

  var %errfile = $PS_TriggersDir $+ TrigChk.txt"
  if (!$isfile(%errfile)) return
  .timerPSTC_ERRCHK off
  window -c @PSTriggerError

  ; resline = chan ; total ; overcount ; no csv ; no report ; 10 day ; bad path ; dupes
  var %resline = $read(%errfile,n,1)
  tokenize 59 %resline

  if ($2 >= 1) {
    ; One path/10d error forgiven for everyone, two or more either kills your leeches or
    ; stops you from opening new leeches (depends on number of errors and credit):
    if ($calc($6 + $7) >= $calc($PSTC_TriggerStats($1).credit / 100 + 1)) set %ps.errorlevel 2
    elseif ($calc($6 + $7) > 1)                                           set %ps.errorlevel 1
    else                                                                  set %ps.errorlevel 0

    if (%nick != $me) .notice %nick @PSChkTrgs; $+ %resline
    PSTC_ShowErrorWindow
  }
  else {
    if (%chanwide) echo $ps_n -st --> Trigger Check found no errors
    else           notice %nick Trigger Check found no errors
    set %ps.errorlevel 0
  }
  .remove %errfile
}

/****************************************************************************************
* Parses and displays trigger error check results. 
*/
on ^*:NOTICE:@PSChkTrgs*:?: {
  haltdef

  tokenize 59 $1-
  var %wname = @PSChkTrgs_ $+ $PS_Network $+ _ $+ $2
  if ($window(%wname) == $null) window -ls -t8,18,28,37,47,56,65 %wname Verdana 11

  aline %wname       $+(03, $nick $chr(9), $&
    04Total Errors: $iif($3 > 0,07,03), $3, $chr(9), $&
    10Overcount:    $iif($4 > 0,07,03), $4, $chr(9), $&
    10No .CSV:      $iif($5 > 0,07,03), $5, $chr(9), $&
    10No Report:    $iif($6 > 0,07,03), $6, $chr(9), $&
    1010 Day:       $iif($7 > 0,07,03), $7, $chr(9), $&
    10Bad Path:     $iif($8 > 0,07,03), $8, $chr(9), $&
    10Duplications: $iif($9 > 0,07,03), $9)
  titlebar %wname --- Errors on $line(%wname, 0) users
  window -b %wname
}

/****************************************************************************************
* Notifies user of problems found during trigger error check. PSTC_PollErrorFile sends
* a reply to the person who requested the check, this pops up a window that displays
* a more detailed list of the errors. This is shown only to the affected user.
* @param void @return void
*/
alias -l PSTC_ShowErrorWindow {
  var %errfile = $PS_TriggersDir(PSTC_ShowErrorWindow) $+ TrigChk.txt"

  var %resline = $read(%errfile, n, 1)
  var %tenday = $gettok(%resline, 6, 59)
  var %badpath = $gettok(%resline, 7, 59)
  var %duplication = $gettok(%resline, 8, 59)

  var %wname = @PSTriggerError
  window -Cla -t9 %wname Verdana 12

  aline %wname $chr(9) 04THIS WINDOW IS SHOWING ERRORS IN YOUR TRIGGERS
  aline %wname $chr(9)
  aline %wname This is a list of problems that were found in your triggers during trigger error check. All users are
  aline %wname expected to keep their triggers in top-notch condition so please fix these errors as soon as possible.
  aline %wname If you don't know how to do this, just ask in channel and someone will (hopefully) advice you.
  aline %wname $chr(9)

  if (%duplication > 0) {
    aline %wname You can fix the duplication errors by downloading and installing latest triggers-update.
    aline %wname Install it using pserve & mirc, the update function in PserveCheck does not fix errors.
    aline %wname $chr(9)
  }

  if (%ps.errorlevel) {
    if (%ps.errorlevel == 2)       aline %wname 04Serious errors found - your leeches have been stopped until they are fixed. Fix these 
    if (%ps.errorlevel == 1)       aline %wname 04Serious errors found - new leech windows are blocked until they are fixed. Fix these 
    aline %wname 04problems and Recheck triggers (right click this window) when you think you've solved
    aline %wname 04the problems. If recheck detects no serious errors it will restore your leech ability.
    if (%badpath > 0)              aline %wname 04- Found %badpath bad paths (non-existant paths or paths containing double spaces).
    if (%tenday > 0)               aline %wname 04- Found %tenday ten-day-rule violations.
  }

  aline %wname $chr(9)
  aline %wname Trigger $chr(9) Error description
  loadbuf 4-99999 %wname %errfile
  window -b %wname
}

menu @PSTriggerError {
  Recheck Triggers: {
    dline $menu 1-99999
    aline $menu $chr(9) | aline $menu $chr(9) | aline $menu $chr(9) | aline $menu $chr(9)
    aline $menu $chr(9) Checking, please wait...
    $PSTC_CheckErrors($me, AllChannels, $true)
  }
}

menu @PSChkTrgs_* {
  Copy To Clipboard: clipboard $replace($sline($menu,1), $chr(9), $chr(32))
  $submenu($PSTL_ViewListSubmenu($1, $strip($gettok($sline($menu,1),1,32)), $chr(35) $+ $gettok($menu,2,35)))
  !PSStatus: ctcp $strip($gettok($sline($menu,1),1,32)) !PSStatus
}



/****************************************************************************************
* Return statistics about configured triggers. Filtering is applied prior to calculating
* the stats, this allows for channel specif stats: <code>$PSTC_TriggerStats(#foo).burned</code>
* and <code>$PSTC_TriggerStats(#bar).burned</code> can thus produce different results. 
* <p>Pserve.exe does the actual job of calculating stats (using the GetStats function).
* The results are cached for 90 seconds, so calling this alias multiple times in rapid
* succession requires only one invocations of pserve.exe. Naturally the cache is cleared
* if the Chan param is different than it was for the previous call. There is no mircscript
* equivalent for GetStats, we are [currently] completely dependant on pserve.exe here.
*
* @param chan Filtering setting, 'AllChannels' to not filter at all.
* @return int number of ____ (specify with property)
* @prop credit     Online final complete triggers, "leechcredit".
* @prop ontrigs    Online triggers
* @prop offtrigs   Offline triggers
* @prop onfiles    Files in online collections
* @prop offfiles   Files in offline collections
* @prop onbytes    Bytes in online collections
* @prop offbytes   Bytes in offline collections
* @prop incomplete Incomplete triggers
* @prop complete   Up-to-date triggers
* @prop overcounts Triggers that have overcounts
* @prop burned     Burned triggers
* @prop unburned   Unburned triggers
* @prop creditbytes   Bytes in up to date final online trigs (new credit)
* @prop unbackuptrigs Number of unbacked up completed trigs
* @prop unbackupbytes Bytes in unbacked up completed trigs
*/
alias PSTC_TriggerStats {
  var %chan = $1

  if ((%ps_trigconf_statscache == $null) || ($gettok(%ps_trigconf_statscache, 01, 59) != %chan)) {
    var %ok = $PS_RunPserveExeSync(GetStats, %chan)
    if (%ok != OK) {
      echo -sat 04Error: Unable to run pserve.exe to calc trigger stats -- %ok
      set -u10 %ps_trigconf_statscache %chan $+ ;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1;-1
    }
    else {
      set -u90 %ps_trigconf_statscache $read($PS_TriggersDir $+ TrigStat.txt", n, 1) 
      .remove $PS_TriggersDir $+ TrigStat.txt"
    }    
  }
  var %stats = %ps_trigconf_statscache

  if ($prop == credit)     return $gettok(%stats, 02, 59)
  if ($prop == ontrigs)    return $gettok(%stats, 03, 59)
  if ($prop == offtrigs)   return $gettok(%stats, 04, 59)
  if ($prop == onfiles)    return $gettok(%stats, 05, 59)
  if ($prop == offfiles)   return $gettok(%stats, 06, 59)
  if ($prop == incomplete) return $gettok(%stats, 07, 59)
  if ($prop == complete)   return $gettok(%stats, 08, 59)
  if ($prop == overcounts) return $gettok(%stats, 09, 59)
  if ($prop == burned)     return $gettok(%stats, 10, 59)
  if ($prop == unburned)   return $gettok(%stats, 11, 59)
  if ($prop == onbytes)    return $gettok(%stats, 12, 59)
  if ($prop == offbytes)   return $gettok(%stats, 13, 59)
  if ($prop == creditbytes) return $gettok(%stats, 14, 59)
  if ($prop == unbackuptrigs) return $gettok(%stats, 15, 59)
  if ($prop == unbackupbytes)  return $gettok(%stats, 16, 59)
  if ($prop == $null)      return %stats
}



/****************************************************************************************
* Does a callback to Alias for every trigger installed on the system. PSTC_Iterate* 
* aliases are similar to mirc's $findfile, which allows you to call an alias for
* every found file.
* <p>
* Iteration can be sync or async, depending on what is needed. Async iteration uses
* a cycle where ~80ms of work is followed by ~20ms sleep. Synched iteration will 
* perform everything in one go, meaning mirc will be frozen for the duration. Use
* async when possible. Regardless of which mode you use, the callback alias MUST NOT
* be async.
* <p>
* If the callback throws an error the iteration will end.
* <p>
* Everything said here also applies to the other PSTC_Iterate* aliases.
* @param alias the callback alias
* @param mode either <code>sync</code> or <code>async</code>
* @param name handle that can be used for cancelling the iteration
* @return void
*/
alias PSTC_IterateAllTrigs {
  var %name = $1, %mode = $2, %alias = $3-, %hname = PSTC_Iteratation_ $+ %name
  hfree -w %hname
  hmake %hname

  hsave PsTriggerArr $PS_TmpDir(PSTC_IterateAllTrigs) $+ TrigArr.hsh"
  hload %hname $PS_TmpDir(PSTC_IterateAllTrigs) $+ TrigArr.hsh"
  .remove $PS_TmpDir(PSTC_IterateAllTrigs) $+ TrigArr.hsh"

  hadd %hname _ITER_ITEMS $hget(%hname, 0).item
  hadd %hname _ITER_ALIAS %alias
  hadd %hname _ITER_DONE 0
  hadd %hname _ITER_MODE %mode
  PSTC_ExecuteIteration %name
}

/****************************************************************************************
* Iteration for all the triggers of a single group.
* @return void
*/ 
alias PSTC_IterateGroupTrigs {
  var %name = $1, %mode = $2, %group = $3, %alias = $4-, %hname = PSTC_Iteratation_ $+ %name
  hfree -w %hname
  hmake %hname

  var %trgfile = $PS_TriggersDir(PSTC_IterateGroupTrigs) $+ %group
  var %i = $ini(%trgfile, 0), %j = 1
  while (%i > 0) {
    var %trig = $ini(%trgfile, %i)
    if (%trig != INFO) { 
      hadd %hname %j %trig
      inc %j
    }
    dec %i
  } 

  hadd %hname _ITER_ITEMS $hget(%hname, 0).item
  hadd %hname _ITER_ALIAS %alias
  hadd %hname _ITER_DONE 0
  hadd %hname _ITER_MODE %mode
  PSTC_ExecuteIteration %name 
}

/****************************************************************************************
* Iterate all groups (really does mean groups, not triggers in those groups)
* @return void
*/ 
alias PSTC_IterateAllGroups {
  var %name = $1, %mode = $2, %alias = $3-, %hname = PSTC_Iteratation_ $+ %name
  hfree -w %hname
  hmake %hname

  var %i = $findfile($PS_TriggersDir(PSTC_IterateAllGroups), *.trg, 0, 0)
  while (%i > 0) {
    hadd %hname %i $upper($nopath($findfile($PS_TriggersDir(PSTC_IterateAllGroups), *.trg, %i, 0)))
    dec %i
  } 

  hadd %hname _ITER_ITEMS $hget(%hname, 0).item
  hadd %hname _ITER_ALIAS %alias
  hadd %hname _ITER_DONE 0
  hadd %hname _ITER_MODE %mode
  PSTC_ExecuteIteration %name 
}

/****************************************************************************************
* This is what actually performs the iteration. @return void
*/
alias -l PSTC_ExecuteIteration {
  var %name = $1, %hname = PSTC_Iteratation_ $+ %name, 
  var %mode = $hget(%hname, _ITER_MODE) 
  var %items = $hget(%hname, _ITER_ITEMS)
  var %alias = $hget(%hname, _ITER_ALIAS)

  var %stoptime = $calc(($ticks + 80) % 2^32)
  ; $ticks wraps at 2^32 - 1, set stoptime at 2^32 and we never reach it
  if (%mode == sync) %stoptime = $calc(2^32)

  while (($ticks < %stoptime) && ($hget(%hname, _ITER_PAUSED) == $null) && ($hget(%hname, _ITER_DONE) < %items)) {
    hinc %hname _ITER_DONE
    %alias $hget(%hname, $hget(%hname, _ITER_DONE))
  }

  if ($hget(%hname, _ITER_PAUSED) != $null) return
  if ($hget(%hname, _ITER_DONE) < %items) {
    ; Sleep a minimum of 20 msec, longer if Alias ran long (aim for 80% CPU utilization)
    .timerPSTC_Iter_ $+ %name -m 1 $int($calc(($ticks - %stoptime) * 0.25 + 20)) PSTC_ExecuteIteration %name
  }
  else {
    hfree %hname
  }
}

/****************************************************************************************
* Control or return information about iteratition whose handle is Name. Without any
* <i>.property</i> this returns the iteration name, if it exists (think of 
* "<b><code>if ($PSTC_Iteration(NAME) != $null)</code></b>" as being equal to
* "<b><code>if ($socket(NAME) != $null)</code></b>" etc).
* @param name iteration handle 
* @prop pause temporarily suspend iteration (return null)
* @prop resume resume suspended iteraration (return null)
* @prop ispaused test whether iteration is paused (return $true/$false)
* @prop stop permanently cancel iteration (return null)
* @prop total return total number of items in this iteration
* @prop progress return number of items processed (alias calls made)
* @prop pc return percent done (rounded down: when .pc hits 100, the very last iteration step has started)
* @return int/name integer number, name of the iteration, or null
*/
alias PSTC_Iteration {
  var %name = $1, %hname = PSTC_Iteratation_ $+ %name

  if ($hget(%hname) == $null) return

  if ($prop == total) {
    return $hget(%hname, _ITER_ITEMS)
  }
  if ($prop == progress) {
    return $hget(%hname, _ITER_DONE)
  }
  if ($prop == pc) {
    return $int($calc($hget(%hname, _ITER_DONE) / $hget(%hname, _ITER_ITEMS) * 100))
  }
  if ($prop == $null) {
    return %name
  }
  if ($prop == ispaused) {
    return $iif($hget(%hname, _ITER_PAUSED) == 1, $true, $false)
  }
  if ($prop == pause) {
    .timerPSTC_Iter_ $+ %name off
    hadd PSTC_Iteratation_ $+ %name _ITER_PAUSED 1
  }
  if ($prop == resume) {
    hdel PSTC_Iteratation_ $+ %name _ITER_PAUSED
    PSTC_ExecuteIteration %name
  }
  if ($prop == stop) {
    .timerPSTC_Iter_ $+ %name off
    hfree -w PSTC_Iteratation_ $+ %name
  }
}



/****************************************************************************************
* Raw read-access to trigger (same as using PS_ReadIni).
* <p>
* Raw access is meant mainly as a development aid. It allows you to easily create and
* access new vars as needed, but once the system is finished new PSTC_Get/Set_____
* aliases should be created and used. This is not a rule set in stone though, raw
* access is still allowed in production code.
* @return string the value of Var or $null if either the Trig or the Var doesn't exist.
*/
alias PSTC_GetRawVar {
  var %trig = $1, %var = $2
  return $PSDB_GetTrig(%trig, %var)
}

/****************************************************************************************
* Raw write-access to trigger (similar to PS_WriteIni). SetRawVar will also trigger
* the psCollectionChanged signal.
* <p>
* Raw access is meant mainly as a development aid. It allows you to easily create and
* access new vars as needed, but once the system is finished new PSTC_Get/Set_____
* aliases should be created and used. This is not a rule set in stone though, raw
* access is still allowed in production code. @optparam value @return void
*/
alias PSTC_SetRawVar {
  var %trig = $1, %var = $2, %value = $3-
  $PSDB_SetTrig(%trig, %var, %value)
}

/****************************************************************************************
* Raw read-access to group's [INFO] section.
* <p>
* Raw access is meant mainly as a development aid. It allows you to easily create and
* access new vars as needed, but (ideally) in production code raw-calls would replaced
* with better abstracted PSTC_* aliases.
* @return string the value of Var or $null if either the Group or the Var doesn't exist.
*/
alias PSTC_GetRawGroupVar {
  var %group = $1, %var = $2
  $PSDB_GetGroup(%group, %var)
}

/****************************************************************************************
* Raw write-access to group's [INFO] section.
* <p>
* Raw access is meant mainly as a development aid. It allows you to easily create and
* access new vars as needed, but (ideally) in production code raw-calls would replaced
* with better abstracted PSTC_* aliases. @optparam value @return void
*/
alias PSTC_SetRawGroupVar {
  var %group = $1, %var = $2, %value = $3-
  $PSDB_SetGroup(%group, %var, %value)
}



/****************************************************************************************
* Access the trigger database to read a value. All PSTC_ aliases access the database
* via this alias, never directly with readini. @return string the value associated 
* with var, or $null
*/
alias -l PSDB_GetTrig {
  var %trig = $1, %var = $2
  if ($hget(PsTriggerMap) == $null) $PSTC_ReloadTriggers
  if ($hget(PsTriggerMap, %trig) == $null) return
  return $remove($readini($PS_TriggersDir(PSDB_GetTrig) $+ $hget(PsTriggerMap, %trig) $+ ", n, %trig, %var),")
}

/****************************************************************************************
* Access the trigger database to write a value. All PSTC_ aliases access the database
* via this alias, never directly with writeini/remini. 
* @optparam value new value for var, omit to unset var @return void
*/
alias -l PSDB_SetTrig {
  var %trig = $1, %var = $2, %value = $3-
  if ($hget(PsTriggerMap) == $null) $PSTC_ReloadTriggers
  if ($hget(PsTriggerMap,%trig) == $null) return
  if (%value == $null) {
    remini -n $PS_TriggersDir(PSTDB_SetTrig) $+ $hget(PsTriggerMap,%trig) $+ " %trig %var
  }
  else {
    writeini -n $PS_TriggersDir(PSTDB_SetTrig) $+ $hget(PsTriggerMap,%trig) $+ " %trig %var $remove(%value, ")
  }
  .signal psCollectionChanged %trig %var $remove(%value, ")
  return
  :error
  echo $ps_k -sat PSDB_SetTrig 04ERROR In: < %trig > < %var > < %value >
  reseterror
  halt
}

/****************************************************************************************
* Access the group database to read a value. All PSTC_ aliases access the database
* via this alias, never directly with readini. @return string the value associated 
* with var, or $null
*/
alias -l PSDB_GetGroup {
  var %group = $1, %var = $2
  return $readini($PS_TriggersDir(PSDB_GetGroup) $+ %group $+ ",n,INFO,%var)
}

/****************************************************************************************
* Access the group database to write a value. All PSTC_ aliases access the database
* via this alias, never directly with writeini/remini. 
* @optparam value new value for var, omit to unset var @return void
*/
alias -l PSDB_SetGroup {
  var %group = $1, %var = $2, %value = $3-
  if (%value == $null) {
    remini -n $PS_TriggersDir(PSTDB_SetGroup) $+ %group $+ " INFO %var
  }
  else {
    writeini -n $PS_TriggersDir(PSTDB_SetGroup) $+ %group $+ " INFO %var %value
  }
  return
  :error
  echo $ps_k -sat PSDB_SetGroup 04ERROR In: < %group > < %var > < %value >
  reseterror
  halt
}





/****************************************************************************************
* Renames empty .trg files to .trm extension and moves them to UnloadedTriggers dir.
* @param void @return void
*/
alias PSTC_PurgeEmptyGroups {
  if (!$isdir($PS_UnloadedTriggersDir $+ $PS.Version.date)) mkdir $PS_UnloadedTriggersDir $+ $PS.Version.date $+ "

  var %trigsbefore = $hget(PsTriggerMap,0).item
  var %groupsbefore = $findfile($PS_TriggersDir, *.trg, 0, 1)
  PSTC_IterateAllGroups PurgeEmptyGroups sync PSTC_UnloadEmptyGroup
  var %trigsafter = $hget(PsTriggerMap,0).item
  var %groupsafter = $findfile($PS_TriggersDir, *.trg, 0, 1)
  echo $ps_n -st --> Unloaded total $+ $ps_kh $calc(%groupsbefore - %groupsafter) groups, containing $+ $ps_kh $calc(%trigsbefore - %trigsafter) triggers
}

/****************************************************************************************
* Helper for PSTC_PurgeEmptyGroups that actually deletes the groups. @return void
*/
alias -l PSTC_UnloadEmptyGroup {
  var %group = $1
  var %trgfile = $PS_TriggersDir(PSTC_DeleteEmptyGroup ) $+ %group $+ "

  if ($read(%trgfile,nw,path=*) != $null) {
    echo $ps_n -st --> Keeping $ps_kh $+ %group $+ , contains configured triggers
  }
  elseif ($PSTC_GetGroupType(%group) == custom) {
    echo $ps_n -st --> Keeping $ps_kh $+ %group $+ , custom groups must be deleted manually
  }
  else {
    var %trmfile = $PS_UnloadedTriggersDir $+ $PS.Version.date $+ \ $+ $left(%group, -4) $+ .trm"
    .rename %trgfile %trmfile
    echo $ps_n -st --> Unloading $ps_kh $+ %group
  }

  ; Reload triggers after all groups have been processed
  if ($PSTC_Iteration(PurgeEmptyGroups).pc == 100) {
    PSTC_ReloadTriggers

    .signal PS2PSC_ReloadTriggers

    ; Instruct PSClite to reload triggers by touching PSTrigRev file
    var %revfile = $PS_TriggersDir(PSTC_RemoveCustomGroup) $+ PSTrigRev.txt"
    var %oldsize = $file(%revfile).size
    write %revfile Just touching the file..
    btrunc %revfile %oldsize
  }
}

/****************************************************************************************
* Calls pserve.exe to update and reload triggers. 
* @param void @return Statsfile
*/
alias PSTC_RestoreEmptyGroups {
  var %scriptdir = $left($PS_ScriptDir, -1) $+ "
  var %trigver   = $PS.Version.date
  var %unloaddir = $PS_UnloadedTriggersDir $+ $PS.Version.date $+ "
  if (!$isdir(%unloaddir)) return

  var %ok = $PS_RunPserveExeSync(RestoreUnloaded, %trigver)
  if (%ok != OK) { 
    echo 4 -sat Photoserve Error: Unable to restore triggers
    echo 4 -sat $+ %ok
    return
  } 

  rmdir %unloaddir
  PSTC_ReloadTriggers $true
  .signal PS2PSC_ReloadTriggers
  return %statsfile
}







/****************************************************************************************
* Calls pserve.exe to update and reload triggers. 
* @param void @return Statsfile
*/
alias PSTC_UpdateTriggers {
  var %scriptdir = $left($PS_ScriptDir, -1) $+ "
  var %trigver   = $PS.Version.date
  var %trmdir    = " $+ $left($mircdir,-1) $+ "
  var %logfile   = $PS_ReadMeDir $+ PserveUpdate.log"
  var %statsfile = $PS_ReadMeDir $+ PserveUpdateStats.txt"
  var %unloaddir = $PS_UnloadedTriggersDir $+ $PS.Version.date $+ "

  var %ok = $PS_RunPserveExeSync(UpdateTriggers, %trigver %trmdir %logfile %statsfile)
  if (%ok != OK) { 
    echo -s 04Photoserve Error: Trigger update failed!!!
    echo -s 07 $+ %ok
    return
  } 
  if ($isdir(%unloaddir)) rmdir %unloaddir


  PSTC_ReloadTriggers $true
  .signal PS2PSC_ReloadTriggers
  return %statsfile
}


/****************************************************************************************
* Rebuilds PsTriggerMap, PsTriggerArr and filterhashes (replaces PS_TriggerMapFlush).
* @optparam dontbuild pass a $true value to only reload existing hashes without rebuilding
* @return void
*/
alias PSTC_ReloadTriggers {
  var %dontbuild = $1

  hfree -w PsTriggerMap
  hfree -w PsTriggerArr
  hmake PsTriggerMap 10000
  hmake PsTriggerArr 10000

  var %triggersdir = $PS_TriggersDir(PSTC_ReloadTriggers)
  if (!%dontbuild) {
    var %result = $PS_RunPserveExeSync(RebuildHashes)
  }
  hload PsTriggerMap %triggersdir $+ TrigMap.hsh"
  hload PsTriggerArr %triggersdir $+ TrigArr.hsh"
  .remove %triggersdir $+ TrigMap.hsh"
  .remove %triggersdir $+ TrigArr.hsh"

  PSF_ReloadFilters
}



/****************************************************************************************
* Mircscript (=slow) version of <code>PSTC_ReloadTriggers</code>. This is just a backup
* in case we run in to problems with pserve.exe, currently we don't use this alias at all.
* NOTE: This does not rebuild or reload filters so we're pretty fucked if the exe breaks
* @param void @return void
*/
alias PSTC_ReloadTriggers_mircscript {
  hfree -w PsTriggerMap
  hfree -w PsTriggerArr
  hmake PsTriggerMap 10000
  hmake PsTriggerArr 10000

  set %ps_trigconf_arritems 1
  var %t = $findfile($PS_TriggersDir, *.trg, 0, 1, PSTC_TriggerMapAdd $1-)
  unset %ps_trigconf_arritems
}

/****************************************************************************************
* Adds the contents of Trgfile to PsTriggerMap and PsTriggerArr. This is only used by
* <code>PSTC_ReloadTriggers_mircscript</code>, so currently disabled.
* @return void
*/
alias -l PSTC_TriggerMapAdd {
  var %trgfile = $1-

  var %i = $ini(%trgfile, 0)
  while (%i > 0) {
    var %trig = $upper($ini(%trgfile, %i))
    if (%trig == INFO) {
      ; no action for the info section
    }
    elseif ($hget(PsTriggerMap,%trig) == $null) {
      hadd PsTriggerMap %trig $upper($nopath(%trgfile))
      hadd PsTriggerArr %ps_trigconf_arritems %trig
      inc %ps_trigconf_arritems
    }
    ; Since this alias is only used if pserve.exe is unavailable, and trigger checks live in pserve.exe, we do some of the checks here:
    elseif ($hget(PsTriggerMap, %trig) == $nopath(%trgfile)) {
      echo $ps_n -sat --> 04Photoserve Warning: $+ $ps_kh Trigger $+ $ps_ki %trig  $+ $ps_kh has become duplicated inside04 $nopath(%trgfile)
    }
    else {
      echo $ps_n -sat --> 04Photoserve Warning: $+ $ps_kh Trigger $+ $ps_ki %trig $+ $ps_kh exists in both04 $nopath(%trgfile) $+ $ps_ki and04 $hget(PsTriggerMap,%trig)
    }
    dec %i
  }
}




;; The old stuff @param void @return void
alias PS_TriggerMapFlush {
  PSTC_ReloadTriggers
}



;; The old stuff
alias PS_ReadIni {
  if (%PS_debug_alias != $null) echo $ps_h -st * PS_ReadIni * $1-
  if ($hget(PsTriggerMap) == $null) $PSTC_ReloadTriggers
  if ($hget(PsTriggerMap,$1) == $null) return
  var %trgfile = $PS_TriggersDir(PS_ReadIni) $+ $hget(PsTriggerMap,$1) $+ "
  if ($2 == findfile) return %trgfile

  var %value = $readini(%trgfile,n,$1,$2)
  if (%value == $null) return
  if ($2 == path) return $replace($+(%value,\),\\,\)
  return %value   
}
